#!/usr/bin/env python3
"""
Balochistan Revenue Authority (BRA) Chatbot Backend
Flask-based API that provides intelligent responses to BRA-related queries
"""

from flask import Flask, request, jsonify, render_template, send_from_directory
from flask_cors import CORS
import re
import random
from datetime import datetime
import json
import os

app = Flask(__name__)
CORS(app)  # Enable CORS for all routes

class BRAKnowledgeBase:
    """Knowledge base containing BRA information and FAQ responses"""
    
    def __init__(self):
        self.bra_info = {
            "introduction": {
                "title": "Balochistan Revenue Authority (BRA)",
                "description": "Welcome to Balochistan Revenue Authority (BRA), the autonomous body responsible for optimizing revenue collection and promoting compliance with tax and related laws in the province of Balochistan, Pakistan.",
                "vision": "To be a modern, progressive, and credible organization, committed to providing quality service and ensuring fair taxation practices.",
                "mission": "To enhance the capability of the tax system through the application of modern techniques and by offering extensive taxpayer assistance.",
                "values": "We are driven by a dedicated and professional workforce, fostering integrity, professionalism, team work, courtesy, fairness, transparency, and responsiveness in all our operations.",
                "goal": "Together, we aim to build a financially sustainable future for Balochistan and its people, supporting economic growth and development for the prosperity of our nation."
            },
            "contact": {
                "phone": "081-2452073",
                "email": "esupport@bra.gob.pk",
                "address": "Balochistan Revenue Authority, Kalat Mall, First Floor, Raisani Road, Quetta",
                "hours": "Mon - Fri 9:00 - 4:00"
            },
            "services": {
                "e_registration": {
                    "name": "e-Registration (أی رجسٹریشن)",
                    "description": "Taxpayers Registration Application",
                    "sub_services": [
                        "New e-Registration Having NTN",
                        "New e-Registration Having No NTN", 
                        "Check Status of Registration Application",
                        "Sign-up for Withholding Agents & Franchisor (Not Having NTN)"
                    ]
                },
                "e_enrollment": {
                    "name": "e-Enrollment (أی أنرولمنٹ)",
                    "description": "Request for e-Enrolment at BRA",
                    "sub_services": [
                        "New Enrollment",
                        "Enrollment Activation"
                    ]
                },
                "e_signup": {
                    "name": "e-Sign Up (أی سأئن أپ)",
                    "description": "Sign-up as Withholding Agent"
                },
                "e_filing": {
                    "name": "e-Filing of Return (أی ریٹرن فأئل)",
                    "description": "File your Return online with ease"
                },
                "e_payment": {
                    "name": "e-Payment of Tax (أی ٹیکس پیمنٹ)",
                    "description": "Internet Base ADC / e-Payment"
                },
                "search_taxpayers": {
                    "name": "Search Taxpayers (سرچ ٹیکس پیئر)",
                    "description": "Taxpayer Online Verification",
                    "sub_services": [
                        "Active Taxpayer List",
                        "Blacklisted/Suspended Taxpayers List",
                        "POS Verification"
                    ]
                }
            },
            "documents": {
                "BST-01": {
                    "name": "Registration Form BST-01",
                    "file": "BST-01_Registration_Form.txt",
                    "description": "Application form for Sales Tax on Services registration"
                },
                "BST-03": {
                    "name": "Return Form BST-03", 
                    "file": "BST-03_Return_Form.txt",
                    "description": "Monthly return form for Sales Tax on Services"
                },
                "BST-04": {
                    "name": "Payment Challan BST-04",
                    "file": "BST-04_Payment_Challan.txt", 
                    "description": "Payment challan form for tax payment"
                }
            },
            "download_categories": {
                "Acts": {
                    "name": "Acts & Legislation",
                    "description": "Legal acts and legislation documents",
                    "url": "https://ebra.com.pk/downloads"
                },
                "Forms": {
                    "name": "Tax Forms",
                    "description": "All tax-related forms including BST-01, BST-03, BST-04",
                    "url": "https://ebra.com.pk/downloads"
                },
                "Rules": {
                    "name": "Rules & Regulations",
                    "description": "Tax rules and regulatory documents",
                    "url": "https://ebra.com.pk/downloads"
                },
                "Notifications_SalesTax": {
                    "name": "Sales Tax Notifications",
                    "description": "Official sales tax notifications and updates",
                    "url": "https://ebra.com.pk/downloads"
                },
                "Notifications_Administration": {
                    "name": "Administrative Notifications",
                    "description": "Administrative notifications and circulars",
                    "url": "https://ebra.com.pk/downloads"
                },
                "Circulars": {
                    "name": "Circulars",
                    "description": "Official circulars and instructions",
                    "url": "https://ebra.com.pk/downloads"
                },
                "NBP_Authorized_Branches": {
                    "name": "NBP Authorized Branches",
                    "description": "List of National Bank authorized branches for tax payment",
                    "url": "https://ebra.com.pk/downloads"
                },
                "Schedules": {
                    "name": "Tax Schedules",
                    "description": "Tax schedules and rate structures",
                    "url": "https://ebra.com.pk/downloads"
                },
                "Service_Sectors": {
                    "name": "Service Sectors",
                    "description": "Information about taxable service sectors",
                    "url": "https://ebra.com.pk/downloads"
                },
                "Publications": {
                    "name": "Publications",
                    "description": "BRA publications and reports",
                    "url": "https://ebra.com.pk/downloads"
                },
                "Appellate_Tribunal": {
                    "name": "Appellate Tribunal",
                    "description": "Appellate tribunal documents and procedures",
                    "url": "https://ebra.com.pk/downloads"
                },
                "Tenders": {
                    "name": "Tenders",
                    "description": "Current and past tender notices",
                    "url": "https://ebra.com.pk/downloads"
                }
            },
            "management": {
                "chairperson": {
                    "name": "Abdullah Khan",
                    "position": "Chairperson",
                    "phone": "081-2452073, Ext: 0",
                    "email": "chairperson@bra.gob.pk"
                },
                "members": [
                    {
                        "name": "Syed Sikandar Shah",
                        "position": "Member Audit",
                        "phone": "081-2452073, Ext: 116",
                        "email": "smember.audit@bra.gob.pk"
                    },
                    {
                        "name": "Hassan Mengal",
                        "position": "Member IT",
                        "phone": "081-2452073, Ext: 0",
                        "email": "member.it@bra.gob.pk"
                    },
                    {
                        "name": "Abdul Rauf",
                        "position": "Member Operations",
                        "phone": "081-2452073, Ext: 117",
                        "email": "abdulrauf.nasir@bra.gob.pk"
                    },
                    {
                        "name": "Ehtisham-ul-Haq",
                        "position": "Member HR",
                        "phone": "081-2452073, Ext: 109",
                        "email": "member.hr@bra.gob.pk"
                    }
                ],
                "it_team": [
                    {
                        "name": "Hassan Mengal",
                        "position": "Member IT",
                        "phone": "081-2452073, Ext: 0",
                        "email": "member.it@bra.gob.pk",
                        "role": "IT Head and overall IT strategy"
                    },
                    {
                        "name": "Sadaqat Khan",
                        "position": "Full Stack Developer",
                        "phone": "081-2452073, Ext: 0",
                        "email": "developer@bra.gob.pk",
                        "role": "Full stack developer currently hired by PRAL but working with BRA IT Team"
                    },
                    {
                        "name": "Muhammad Shoaib",
                        "position": "System Administrator",
                        "phone": "081-2452073, Ext: 0",
                        "email": "sysadmin@bra.gob.pk",
                        "role": "System administration and IT infrastructure management"
                    }
                ],
                "commissioners": [
                    {
                        "name": "Yaqoob Khan Marri",
                        "position": "Commissioner",
                        "phone": "081-2452073, Ext: 113",
                        "email": "commissionerops@bra.gob.pk"
                    },
                    {
                        "name": "Shahnawaz Nousherwani",
                        "position": "Commissioner (Audit)",
                        "phone": "081-2452073, Ext: 112",
                        "email": "commissioner.audit@bra.gob.pk"
                    },
                    {
                        "name": "Fahad Shabbir",
                        "position": "Commissioner-II",
                        "phone": "081-2452073, Ext: 118",
                        "email": "fahad.shabbir@bra.gob.pk"
                    },
                    {
                        "name": "Zahid Ahmed Kasi",
                        "position": "Commissioner-III",
                        "phone": "081-2452073, Ext: 115",
                        "email": "zahid.ahmed@bra.gob.pk"
                    }
                ],
                "assistant_commissioners": [
                    {
                        "name": "Jahangir Khan",
                        "position": "Assistant Commissioner",
                        "phone": "081-2452073, Ext: 118",
                        "email": "Jahangirfbr18@gmail.com"
                    },
                    {
                        "name": "Saqib Siddique",
                        "position": "Assistant Commissioner",
                        "phone": "081-2452073, Ext: 0",
                        "email": "saqib.siddique@bra.gob.pk"
                    },
                    {
                        "name": "Muhammad Waqar Mushtaq",
                        "position": "Assistant Commissioner",
                        "phone": "081-2452073, Ext: 0",
                        "email": "waqar.mushtaq@bra.gob.pk"
                    },
                    {
                        "name": "Agha Shahar Yar",
                        "position": "Assistant Commissioner",
                        "phone": "081-2452073, Ext: 0",
                        "email": "aghashahar.yar@bra.gob.pk"
                    },
                    {
                        "name": "Ismail Sanjrani",
                        "position": "Assistant Commissioner",
                        "phone": "081-2452073, Ext: 120",
                        "email": "ismail.sanjrani@bra.gob.pk"
                    },
                    {
                        "name": "Hamza Ali Baloch",
                        "position": "Assistant Commissioner",
                        "phone": "081-2452073, Ext: 0",
                        "email": "hamza.ali@bra.gob.pk"
                    },
                    {
                        "name": "Azizullah Kakar",
                        "position": "Assistant Commissioner",
                        "phone": "081-2452073, Ext: 0",
                        "email": "aziz.ullah@bra.gob.pk"
                    },
                    {
                        "name": "Muhammad Jan",
                        "position": "Assistant Commissioner",
                        "phone": "081-2452073, Ext: 0",
                        "email": "muhammad.jan@bra.gob.pk"
                    },
                    {
                        "name": "Siraj-ud-Din",
                        "position": "Assistant Commissioner",
                        "phone": "081-2452073, Ext: 0",
                        "email": "siraj.uddin@bra.gob.pk"
                    },
                    {
                        "name": "Obaid Ullah",
                        "position": "Assistant Commissioner",
                        "phone": "081-2452073, Ext: 118",
                        "email": "obaidgmbaloch@gmail.com"
                    },
                    {
                        "name": "Muhammad Yaqoob Sanjrani",
                        "position": "Assistant Commissioner",
                        "phone": "081-2452073, Ext: 120",
                        "email": "Yaqoobsanjarani@gmail.com"
                    }
                ],
                "other_staff": [
                    {
                        "name": "Muhammad Shoaib",
                        "position": "System Administrator",
                        "phone": "081-2452073, Ext: 0",
                        "email": "sysadmin@bra.gob.pk"
                    },
                    {
                        "name": "Muhammad Babar",
                        "position": "Assistant Manager (Admin)",
                        "phone": "081-2452073, Ext: 0",
                        "email": "muhammad.babar@bra.gob.pk"
                    }
                ]
            },
            "infrastructure_cess": {
                "act_name": "Balochistan Infrastructure Cess Act, 2021 (Amended 2024)",
                "purpose": "Collection of cess on goods transported in, out, or through Balochistan for infrastructure development",
                "coverage": "Entire province of Balochistan - goods by road, rail, air, sea, or pipelines",
                "rates": {
                    "general": "1.10% to 1.15% of value plus per kilometer fee",
                    "gold": "0.125% of value",
                    "exports": "0% (goods leaving Balochistan for export)"
                },
                "fund_usage": [
                    "Roads and bridges development",
                    "Public parks and facilities",
                    "Hospitals and healthcare facilities", 
                    "Schools and educational infrastructure",
                    "Streetlights and safety services",
                    "General infrastructure maintenance"
                ],
                "penalties": "Equal to evaded amount, goods/vehicle seizure authority",
                "appeals": "30 days for appeals, 3 months for refund claims"
            }
        }
        
        # Comprehensive FAQ database from BRA website
        self.faqs = [
            {
                "question": "What is BRA?",
                "answer": "Balochistan Revenue Authority (BRA) is the autonomous body responsible for optimizing revenue collection and promoting compliance with tax and related laws in the province of Balochistan, Pakistan. We are committed to providing quality service and ensuring fair taxation practices.",
                "keywords": ["what is bra", "bra", "balochistan revenue authority", "introduction", "about bra"]
            },
            {
                "question": "What is the introduction of BRA?",
                "answer": "Welcome to Balochistan Revenue Authority (BRA), the autonomous body responsible for optimizing revenue collection and promoting compliance with tax and related laws in the province of Balochistan, Pakistan. Our vision is to be a modern, progressive, and credible organization, committed to providing quality service and ensuring fair taxation practices.",
                "keywords": ["introduction", "bra introduction", "about bra", "welcome", "what is bra"]
            },
            {
                "question": "Tell me about BRA",
                "answer": "Balochistan Revenue Authority (BRA) is the autonomous body responsible for optimizing revenue collection and promoting compliance with tax and related laws in the province of Balochistan, Pakistan. Our mission is to enhance the capability of the tax system through the application of modern techniques and by offering extensive taxpayer assistance.",
                "keywords": ["tell me about bra", "about bra", "bra information", "bra details"]
            },
            {
                "question": "What are BRA services?",
                "answer": "BRA offers comprehensive e-Services including e-Registration (أی رجسٹریشن) for taxpayer registration, e-Enrollment (أی أنرولمنٹ) for BRA enrollment, e-Sign Up (أی سأئن أپ) for withholding agents, e-Filing of Return (أی ریٹرن فأئل) for online return filing, e-Payment of Tax (أی ٹیکس پیمنٹ) for online payments, and Search Taxpayers (سرچ ٹیکس پیئر) for verification services.",
                "keywords": ["bra services", "services", "e-services", "what services", "bra offers"]
            },
            {
                "question": "How can I register with BRA?",
                "answer": "You can register with BRA through our e-Registration service. We offer New e-Registration for those with NTN, New e-Registration for those without NTN, and you can check the status of your registration application online.",
                "keywords": ["register", "registration", "how to register", "e-registration", "new registration"]
            },
            {
                "question": "How to pay taxes to BRA?",
                "answer": "You can pay taxes to BRA through our e-Payment of Tax service, which provides Internet-based ADC and e-Payment options for convenient online tax payments.",
                "keywords": ["pay taxes", "payment", "e-payment", "how to pay", "tax payment"]
            },
            {
                "question": "How to file returns with BRA?",
                "answer": "You can file your returns with BRA through our e-Filing of Return service, which allows you to file your return online with ease.",
                "keywords": ["file return", "return filing", "e-filing", "how to file", "returns"]
            },
            {
                "question": "What does BRA stand for?",
                "answer": "BRA stands for Balochistan Revenue Authority. It is the autonomous body responsible for optimizing revenue collection and promoting compliance with tax and related laws in the province of Balochistan, Pakistan.",
                "keywords": ["what does bra stand for", "bra stands for", "bra meaning", "full form", "abbreviation"]
            },
            {
                "question": "What is Balochistan Revenue Authority?",
                "answer": "Balochistan Revenue Authority (BRA) is the autonomous body responsible for optimizing revenue collection and promoting compliance with tax and related laws in the province of Balochistan, Pakistan. Our vision is to be a modern, progressive, and credible organization, committed to providing quality service and ensuring fair taxation practices.",
                "keywords": ["balochistan revenue authority", "what is balochistan revenue authority", "bra organization", "revenue authority"]
            },
            {
                "question": "Where is BRA located?",
                "answer": "BRA is located at Balochistan Revenue Authority, Kalat Mall, First Floor, Raisani Road, Quetta. Our office hours are Monday to Friday, 9:00 AM to 4:00 PM.",
                "keywords": ["where is bra", "bra location", "bra address", "bra office", "quetta office"]
            },
            {
                "question": "How to contact BRA?",
                "answer": "You can contact BRA through:\n📞 Phone: 081-2452073\n📧 Email: esupport@bra.gob.pk\n📍 Address: Balochistan Revenue Authority, Kalat Mall, First Floor, Raisani Road, Quetta\n🕒 Hours: Mon - Fri 9:00 - 4:00",
                "keywords": ["contact bra", "bra contact", "phone number", "email", "how to contact", "bra support"]
            },
            {
                "question": "What are BRA office hours?",
                "answer": "BRA office hours are Monday to Friday, 9:00 AM to 4:00 PM. We are closed on weekends and public holidays.",
                "keywords": ["office hours", "bra hours", "working hours", "when is bra open", "business hours"]
            },
            {
                "question": "What is BRA website?",
                "answer": "BRA official website is ebra.com.pk. You can access all e-Services, download forms, and get information about BRA services through our website.",
                "keywords": ["bra website", "website", "ebra.com.pk", "online portal", "bra portal"]
            },
            {
                "question": "How to register for BRA services?",
                "answer": "You can register for BRA services through our e-Registration portal at ebra.com.pk. We offer New e-Registration for those with NTN, New e-Registration for those without NTN, and you can check the status of your registration application online.",
                "keywords": ["register", "registration", "how to register", "bra registration", "sign up"]
            },
            {
                "question": "What is Sales Tax on Services?",
                "answer": "Sales Tax on Services is an indirect levy applicable to services specified in Second Schedule on 'taxable services'. The rate is 15% except telecommunication where it is 19.5% on the 'value of taxable services'.",
                "keywords": ["sales tax on services", "what is sales tax", "tax on services", "bsts", "service tax"]
            },
            {
                "question": "What is the rate of Sales Tax on Services?",
                "answer": "Rate of Sales Tax on Services is 15% except telecommunication where it is 19.5% on the 'value of taxable services'.",
                "keywords": ["tax rate", "sales tax rate", "15%", "19.5%", "telecommunication rate", "how much tax"]
            },
            {
                "question": "How to pay BRA taxes online?",
                "answer": "You can pay BRA taxes online through our e-Payment of Tax service, which provides Internet-based ADC and e-Payment options for convenient online tax payments. Visit ebra.com.pk to access the payment portal.",
                "keywords": ["pay taxes online", "online payment", "e-payment", "tax payment", "how to pay", "payment methods"]
            },
            {
                "question": "What forms are available for download?",
                "answer": "BRA provides various forms for download including BST-01 Registration Form, BST-03 Return Form, and BST-04 Payment Challan. You can download these forms from our website or ask me to provide download links.",
                "keywords": ["forms", "download forms", "bst forms", "registration form", "return form", "challan form"]
            },
            {
                "question": "What is Infrastructure Cess?",
                "answer": "Infrastructure Cess is a tax collected on goods transported in, out, or through Balochistan by road, rail, air, sea, or pipelines. The revenue is used exclusively for the development and maintenance of provincial infrastructure like roads, bridges, hospitals, and schools.",
                "keywords": ["infrastructure cess", "cess", "infrastructure tax", "balochistan cess", "what is cess"]
            },
            {
                "question": "What is BRA vision?",
                "answer": "BRA's vision is to be a modern, progressive, and credible organization, committed to providing quality service and ensuring fair taxation practices.",
                "keywords": ["bra vision", "vision", "bra goals", "organization vision"]
            },
            {
                "question": "What is BRA mission?",
                "answer": "BRA's mission is to enhance the capability of the tax system through the application of modern techniques and by offering extensive taxpayer assistance.",
                "keywords": ["bra mission", "mission", "bra purpose", "organization mission"]
            },
            {
                "question": "Who is the BRA Chairperson?",
                "answer": "Mr. Abdullah Khan is the current Chairperson of Balochistan Revenue Authority. He was transferred and posted as Chairperson BRA w.e.f. 19th June, 2025.",
                "keywords": ["chairperson", "chairman", "bra head", "who is chairperson", "bra leader"]
            },
            {
                "question": "What is BRA management structure?",
                "answer": "BRA is led by Chairperson Mr. Abdullah Khan, with key management including IT Member, HR Member, Audit Member, Operations Member, and various Commissioners responsible for different regions and functions.",
                "keywords": ["management", "bra management", "leadership", "organization structure", "bra team"]
            },
            {
                "question": "Who are the BRA commissioners?",
                "answer": "BRA has four Commissioners: Yaqoob Khan Marri (Commissioner), Shahnawaz Nousherwani (Commissioner Audit), Fahad Shabbir (Commissioner-II), and Zahid Ahmed Kasi (Commissioner-III). Each handles specific operational areas and regions.",
                "keywords": ["commissioners", "bra commissioners", "commissioner list", "who are commissioners", "commissioner details"]
            },
            {
                "question": "Who are the BRA assistant commissioners?",
                "answer": "BRA has 11 Assistant Commissioners: Jahangir Khan, Saqib Siddique, Muhammad Waqar Mushtaq, Agha Shahar Yar, Ismail Sanjrani, Hamza Ali Baloch, Azizullah Kakar, Muhammad Jan, Siraj-ud-Din, Obaid Ullah, and Muhammad Yaqoob Sanjrani. They assist in various administrative and operational functions.",
                "keywords": ["assistant commissioners", "bra assistant commissioners", "assistant commissioner list", "who are assistant commissioners", "ac details"]
            },
            {
                "question": "Who is the BRA Chairperson?",
                "answer": "Mr. Abdullah Khan is the current Chairperson of Balochistan Revenue Authority. He was transferred and posted as Chairperson BRA w.e.f. 19th June, 2025. Contact: 081-2452073, Ext: 0, chairperson@bra.gob.pk",
                "keywords": ["chairperson", "chairman", "bra head", "who is chairperson", "bra leader", "abdullah khan"]
            },
            {
                "question": "Who is responsible for audit at BRA?",
                "answer": "Mr. Syed Sikandar Shah is the Member Audit at Balochistan Revenue Authority. You can contact him at smember.audit@bra.gob.pk or 081-2452073, Ext: 116.",
                "keywords": ["audit member", "member audit", "syed sikandar shah", "audit department", "who handles audit"]
            },
            {
                "question": "Who is the IT Member at BRA?",
                "answer": "Hassan Mengal is the Member IT at Balochistan Revenue Authority. He handles all IT and technology-related matters. Contact: 081-2452073, Ext: 0, member.it@bra.gob.pk",
                "keywords": ["it member", "member it", "hassan mengal", "it department", "technology head", "it head"]
            },
            {
                "question": "Who is the HR Member at BRA?",
                "answer": "Ehtisham-ul-Haq is the Member HR at Balochistan Revenue Authority. He handles human resources and personnel matters. Contact: 081-2452073, Ext: 109, member.hr@bra.gob.pk",
                "keywords": ["hr member", "member hr", "ehtisham ul haq", "hr department", "human resources", "hr head"]
            },
            {
                "question": "Who is the Operations Member at BRA?",
                "answer": "Abdul Rauf is the Member Operations at Balochistan Revenue Authority. He handles operational matters and day-to-day activities. Contact: 081-2452073, Ext: 117, abdulrauf.nasir@bra.gob.pk",
                "keywords": ["operations member", "member operations", "abdul rauf", "operations department", "operations head"]
            },
            {
                "question": "Who is the System Administrator at BRA?",
                "answer": "Muhammad Shoaib is the System Administrator at Balochistan Revenue Authority. He manages technical systems and IT infrastructure. Contact: 081-2452073, Ext: 0, sysadmin@bra.gob.pk",
                "keywords": ["system administrator", "sysadmin", "muhammad shoaib", "technical admin", "system admin"]
            },
            {
                "question": "Who is the Assistant Manager Admin at BRA?",
                "answer": "Muhammad Babar is the Assistant Manager (Admin) at Balochistan Revenue Authority. He handles administrative matters and support functions. Contact: 081-2452073, Ext: 0, muhammad.babar@bra.gob.pk",
                "keywords": ["assistant manager admin", "admin manager", "muhammad babar", "administrative manager", "admin head"]
            },
            {
                "question": "Who is the IT team at BRA?",
                "answer": "BRA IT Team consists of Hassan Mengal (Member IT - IT Head), Sadaqat Khan (Full Stack Developer), and Muhammad Shoaib (System Administrator). They handle all IT operations, system development, and technical infrastructure at BRA.",
                "keywords": ["it team", "bra it team", "it department", "it staff", "technology team", "it members", "it personnel"]
            },
            {
                "question": "Who is the IT Head at BRA?",
                "answer": "Hassan Mengal is the Member IT and IT Head at Balochistan Revenue Authority. He leads the IT strategy and oversees all technology-related matters. Contact: 081-2452073, Ext: 0, member.it@bra.gob.pk",
                "keywords": ["it head", "it leader", "hassan mengal", "member it", "it director", "it chief"]
            },
            {
                "question": "Who is the Full Stack Developer at BRA?",
                "answer": "Sadaqat Khan is the Full Stack Developer at BRA. He is currently hired by PRAL but works with the BRA IT Team on system development and technical projects. Contact: 081-2452073, Ext: 0, developer@bra.gob.pk",
                "keywords": ["full stack developer", "developer", "sadaqat khan", "programmer", "software developer", "it developer"]
            },
            {
                "question": "Who is the System Administrator at BRA?",
                "answer": "Muhammad Shoaib is the System Administrator at BRA. He manages technical systems, IT infrastructure, and handles system administration tasks. Contact: 081-2452073, Ext: 0, sysadmin@bra.gob.pk",
                "keywords": ["system administrator", "sysadmin", "muhammad shoaib", "technical admin", "system admin", "infrastructure admin"]
            },
            {
                "question": "What does the IT team do at BRA?",
                "answer": "The BRA IT Team handles all technology operations including system development, IT infrastructure management, technical support, and digital services. The team consists of Hassan Mengal (IT Head), Sadaqat Khan (Full Stack Developer), and Muhammad Shoaib (System Administrator).",
                "keywords": ["what does it team do", "it team responsibilities", "it team functions", "it team duties", "it operations"]
            },
            {
                "question": "Who handles IT support at BRA?",
                "answer": "The BRA IT Team provides IT support. Contact Hassan Mengal (Member IT) for IT strategy, Sadaqat Khan (Full Stack Developer) for development issues, or Muhammad Shoaib (System Administrator) for system-related support. All can be reached at 081-2452073, Ext: 0.",
                "keywords": ["it support", "technical support", "it help", "system support", "it assistance", "tech support"]
            },
            {
                "question": "How to check taxpayer status?",
                "answer": "You can check taxpayer status through our Search Taxpayers service at ebra.com.pk. This includes Active Taxpayer List, Blacklisted/Suspended Taxpayers List, and POS Verification services.",
                "keywords": ["check taxpayer", "taxpayer status", "active taxpayer", "verify taxpayer", "taxpayer list"]
            },
            {
                "question": "What is BRA e-Services?",
                "answer": "BRA e-Services include e-Registration, e-Enrollment, e-Sign Up, e-Filing of Return, e-Payment of Tax, and Search Taxpayers. These services provide convenient online access to all tax-related activities.",
                "keywords": ["e-services", "online services", "digital services", "bra portal", "electronic services"]
            },
            {
                "question": "How to become a withholding agent?",
                "answer": "You can become a withholding agent by using our e-Sign Up service. This allows you to sign up as a Withholding Agent and access related services through the BRA portal.",
                "keywords": ["withholding agent", "become agent", "agent registration", "withholding", "sign up agent"]
            },
            {
                "question": "Explain everything about BRA",
                "answer": "## 🏛️ **Complete Guide to Balochistan Revenue Authority (BRA)**\n\n### **📋 What is BRA?**\nBalochistan Revenue Authority (BRA) is the autonomous body responsible for optimizing revenue collection and promoting compliance with tax and related laws in the province of Balochistan, Pakistan.\n\n### **🎯 Our Vision**\nTo be a modern, progressive, and credible organization, committed to providing quality service and ensuring fair taxation practices.\n\n### **🚀 Our Mission**\nTo enhance the capability of the tax system through the application of modern techniques and by offering extensive taxpayer assistance.\n\n### **💎 Our Values**\nWe are driven by a dedicated and professional workforce, fostering integrity, professionalism, team work, courtesy, fairness, transparency, and responsiveness in all our operations.\n\n### **🌟 Our Goal**\nTogether, we aim to build a financially sustainable future for Balochistan and its people, supporting economic growth and development for the prosperity of our nation.\n\n### **📞 Contact Information**\n- **Phone:** 081-2452073\n- **Email:** esupport@bra.gob.pk\n- **Address:** Balochistan Revenue Authority, Kalat Mall, First Floor, Raisani Road, Quetta\n- **Hours:** Mon - Fri 9:00 - 4:00\n- **Website:** ebra.com.pk\n\n### **🛠️ Our e-Services**\n- **e-Registration (أی رجسٹریشن)** - Taxpayers Registration Application\n- **e-Enrollment (أی أنرولمنٹ)** - Request for e-Enrolment at BRA\n- **e-Sign Up (أی سأئن أپ)** - Sign-up as Withholding Agent\n- **e-Filing of Return (أی ریٹرن فأئل)** - File your Return online with ease\n- **e-Payment of Tax (أی ٹیکس پیمنٹ)** - Internet Base ADC / e-Payment\n- **Search Taxpayers (سرچ ٹیکس پیئر)** - Taxpayer Online Verification\n\n### **💰 Tax Information**\n- **Sales Tax on Services:** 15% (19.5% for telecommunication)\n- **Infrastructure Cess:** 1.10% to 1.15% of goods value\n- **Effective Date:** July 01, 2015\n\n### **📄 Available Forms**\n- BST-01: Registration Form\n- BST-03: Return Form\n- BST-04: Payment Challan\n\n### **👥 Management**\n- **Chairperson:** Mr. Abdullah Khan (w.e.f. 19th June, 2025)\n- **Key Members:** IT, HR, Audit, Operations\n- **Commissioners:** Regional and functional responsibilities\n\n### **🏗️ Infrastructure Cess Act 2021**\n- Covers goods transported in/out/through Balochistan\n- Revenue used for infrastructure development\n- Rates: 1.10%-1.15% + per km fee\n- Gold: 0.125% of value\n- Export goods: 0%\n\n### **📱 How to Access Services**\n1. Visit ebra.com.pk\n2. Register for e-Services\n3. Use online portals for payments, returns, and verification\n4. Download forms as needed\n5. Contact BRA for assistance\n\nFor more detailed information about any specific service, feel free to ask me!",
                "keywords": ["explain everything about bra", "tell me everything about bra", "complete bra information", "bra overview", "bra guide", "everything about bra", "bra summary", "bra details", "bra information", "bra explanation"]
            },
            {
                "question": "Tell me everything about BRA",
                "answer": "## 🏛️ **Complete Guide to Balochistan Revenue Authority (BRA)**\n\n### **📋 What is BRA?**\nBalochistan Revenue Authority (BRA) is the autonomous body responsible for optimizing revenue collection and promoting compliance with tax and related laws in the province of Balochistan, Pakistan.\n\n### **🎯 Our Vision**\nTo be a modern, progressive, and credible organization, committed to providing quality service and ensuring fair taxation practices.\n\n### **🚀 Our Mission**\nTo enhance the capability of the tax system through the application of modern techniques and by offering extensive taxpayer assistance.\n\n### **💎 Our Values**\nWe are driven by a dedicated and professional workforce, fostering integrity, professionalism, team work, courtesy, fairness, transparency, and responsiveness in all our operations.\n\n### **🌟 Our Goal**\nTogether, we aim to build a financially sustainable future for Balochistan and its people, supporting economic growth and development for the prosperity of our nation.\n\n### **📞 Contact Information**\n- **Phone:** 081-2452073\n- **Email:** esupport@bra.gob.pk\n- **Address:** Balochistan Revenue Authority, Kalat Mall, First Floor, Raisani Road, Quetta\n- **Hours:** Mon - Fri 9:00 - 4:00\n- **Website:** ebra.com.pk\n\n### **🛠️ Our e-Services**\n- **e-Registration (أی رجسٹریشن)** - Taxpayers Registration Application\n- **e-Enrollment (أی أنرولمنٹ)** - Request for e-Enrolment at BRA\n- **e-Sign Up (أی سأئن أپ)** - Sign-up as Withholding Agent\n- **e-Filing of Return (أی ریٹرن فأئل)** - File your Return online with ease\n- **e-Payment of Tax (أی ٹیکس پیمنٹ)** - Internet Base ADC / e-Payment\n- **Search Taxpayers (سرچ ٹیکس پیئر)** - Taxpayer Online Verification\n\n### **💰 Tax Information**\n- **Sales Tax on Services:** 15% (19.5% for telecommunication)\n- **Infrastructure Cess:** 1.10% to 1.15% of goods value\n- **Effective Date:** July 01, 2015\n\n### **📄 Available Forms**\n- BST-01: Registration Form\n- BST-03: Return Form\n- BST-04: Payment Challan\n\n### **👥 Management**\n- **Chairperson:** Mr. Abdullah Khan (w.e.f. 19th June, 2025)\n- **Key Members:** IT, HR, Audit, Operations\n- **Commissioners:** Regional and functional responsibilities\n\n### **🏗️ Infrastructure Cess Act 2021**\n- Covers goods transported in/out/through Balochistan\n- Revenue used for infrastructure development\n- Rates: 1.10%-1.15% + per km fee\n- Gold: 0.125% of value\n- Export goods: 0%\n\n### **📱 How to Access Services**\n1. Visit ebra.com.pk\n2. Register for e-Services\n3. Use online portals for payments, returns, and verification\n4. Download forms as needed\n5. Contact BRA for assistance\n\nFor more detailed information about any specific service, feel free to ask me!",
                "keywords": ["tell me everything about bra", "complete bra information", "bra overview", "bra guide", "everything about bra", "bra summary", "bra details", "bra information", "bra explanation", "explain bra"]
            },
            {
                "question": "What is BRA overview?",
                "answer": "## 🏛️ **BRA Overview - Balochistan Revenue Authority**\n\n### **📋 What is BRA?**\nBalochistan Revenue Authority (BRA) is the autonomous body responsible for optimizing revenue collection and promoting compliance with tax and related laws in the province of Balochistan, Pakistan.\n\n### **🎯 Our Vision**\nTo be a modern, progressive, and credible organization, committed to providing quality service and ensuring fair taxation practices.\n\n### **🚀 Our Mission**\nTo enhance the capability of the tax system through the application of modern techniques and by offering extensive taxpayer assistance.\n\n### **📞 Contact Information**\n- **Phone:** 081-2452073\n- **Email:** esupport@bra.gob.pk\n- **Address:** Balochistan Revenue Authority, Kalat Mall, First Floor, Raisani Road, Quetta\n- **Hours:** Mon - Fri 9:00 - 4:00\n- **Website:** ebra.com.pk\n\n### **🛠️ Our e-Services**\n- **e-Registration** - Taxpayers Registration Application\n- **e-Enrollment** - Request for e-Enrolment at BRA\n- **e-Sign Up** - Sign-up as Withholding Agent\n- **e-Filing of Return** - File your Return online with ease\n- **e-Payment of Tax** - Internet Base ADC / e-Payment\n- **Search Taxpayers** - Taxpayer Online Verification\n\n### **💰 Tax Information**\n- **Sales Tax on Services:** 15% (19.5% for telecommunication)\n- **Infrastructure Cess:** 1.10% to 1.15% of goods value\n\n### **📄 Available Forms**\n- BST-01: Registration Form\n- BST-03: Return Form\n- BST-04: Payment Challan\n\nFor more detailed information about any specific service, feel free to ask me!",
                "keywords": ["bra overview", "overview", "bra summary", "bra guide", "bra introduction", "bra basics"]
            },
            {
                "question": "BRA guide for beginners",
                "answer": "## 🏛️ **BRA Guide for Beginners - Balochistan Revenue Authority**\n\n### **📋 What is BRA?**\nBalochistan Revenue Authority (BRA) is the autonomous body responsible for optimizing revenue collection and promoting compliance with tax and related laws in the province of Balochistan, Pakistan.\n\n### **🎯 What does BRA do?**\n- Collects Sales Tax on Services (15% rate)\n- Manages Infrastructure Cess (1.10%-1.15% rate)\n- Provides e-Services for taxpayers\n- Maintains taxpayer records\n- Enforces tax compliance\n\n### **📞 How to Contact BRA?**\n- **Phone:** 081-2452073\n- **Email:** esupport@bra.gob.pk\n- **Address:** Balochistan Revenue Authority, Kalat Mall, First Floor, Raisani Road, Quetta\n- **Hours:** Mon - Fri 9:00 - 4:00\n- **Website:** ebra.com.pk\n\n### **🛠️ How to Use BRA Services?**\n1. **Visit ebra.com.pk**\n2. **Register for e-Services**\n3. **Use online portals for:**\n   - Tax payments\n   - Return filing\n   - Registration\n   - Verification\n\n### **📄 What Forms are Available?**\n- **BST-01:** Registration Form\n- **BST-03:** Return Form\n- **BST-04:** Payment Challan\n\n### **💰 What Taxes does BRA Collect?**\n- **Sales Tax on Services:** 15% (19.5% for telecommunication)\n- **Infrastructure Cess:** 1.10% to 1.15% of goods value\n\n### **🏗️ What is Infrastructure Cess?**\n- Tax on goods transported in/out/through Balochistan\n- Revenue used for infrastructure development\n- Rates: 1.10%-1.15% + per km fee\n\n### **📱 How to Get Started?**\n1. Visit ebra.com.pk\n2. Register for e-Services\n3. Download required forms\n4. Make payments online\n5. File returns electronically\n\n### **❓ Need Help?**\n- Contact BRA directly\n- Use our chatbot for quick answers\n- Visit ebra.com.pk for detailed guides\n\nFor more specific information about any service, feel free to ask me!",
                "keywords": ["bra guide for beginners", "beginner guide", "bra basics", "how to use bra", "bra tutorial", "bra help", "bra guide"]
            },
            {
                "question": "What is the effective date of the enforcement of the Act?",
                "answer": "Provision of Sales Tax on Services Act applies from July 01, 2015.",
                "keywords": ["effective date", "enforcement", "act", "when", "start"]
            },
            {
                "question": "Who is authorized to administer the Sales Tax on Services in Province?",
                "answer": "Balochistan Revenue Authority (BRA) is authorized to administer the Sales Tax on Services.",
                "keywords": ["authorized", "administer", "who", "authority", "responsible"]
            },
            {
                "question": "What is Sales Tax on Service?",
                "answer": "Sales Tax on Services is an indirect levy. This tax is applicable to services specified in Second Schedule on 'taxable services'.",
                "keywords": ["what is", "sales tax", "service", "indirect tax", "levy"]
            },
            {
                "question": "What is the rate of Sales Tax on Services?",
                "answer": "Rate of Sales Tax on Services is 15% except telecommunication where it is 19.5% on the 'value of taxable services'.",
                "keywords": ["rate", "percentage", "15%", "19.5%", "telecommunication", "how much"]
            },
            {
                "question": "Who is liable to pay Sales Tax on Services?",
                "answer": "The person who provides the taxable service is responsible for collection and paying the Sales Tax on Services to the Government.",
                "keywords": ["liable", "pay", "responsible", "service provider", "who pays"]
            },
            {
                "question": "What is the procedure for registration?",
                "answer": "A prospective taxable service provider seeking registration should file an application in prescribed form BST-01 (online) within thirty days from the date of notification of the taxable service. The Authority issues Provisional Registration Certificate within three working days online for three months.",
                "keywords": ["registration", "procedure", "how to register", "BST-01", "application"]
            },
            {
                "question": "How and where to pay Sales Tax on Services?",
                "answer": "Sales Tax on Services is paid using Challan Form BST-04 in authorized branches of National Bank. Head of account 'B-02387 - Balochistan Sales Tax on Services' should be used.",
                "keywords": ["how to pay", "where to pay", "payment", "BST-04", "national bank", "challan"]
            },
            {
                "question": "What are BRA office hours?",
                "answer": "BRA office hours are Monday to Friday, 9:00 AM to 4:00 PM.",
                "keywords": ["office hours", "timing", "working hours", "when open"]
            },
            {
                "question": "How can I contact BRA?",
                "answer": "You can contact BRA at:\nPhone: 081-2452073\nEmail: esupport@bra.gob.pk\nAddress: Balochistan Revenue Authority, Kalat Mall, First Floor, Raisani Road, Quetta",
                "keywords": ["contact", "phone", "email", "address", "reach"]
            },
            {
                "question": "What e-services does BRA provide?",
                "answer": "BRA provides the following e-services: e-Registration, e-Enrollment, e-Sign Up, e-Filing of Return, e-Payment of Tax, Search Taxpayers, and POS Verification.",
                "keywords": ["e-services", "online services", "digital services", "services"]
            },
            {
                "question": "Is there any exemption threshold?",
                "answer": "Section 12 provides for exemption, specified rate or reduced rate on case to case basis.",
                "keywords": ["exemption", "threshold", "exempt", "minimum"]
            },
            {
                "question": "Are there any exempt services?",
                "answer": "All services outside of Second Schedule are exempt from Sales Tax on Services.",
                "keywords": ["exempt services", "exemption", "not taxable"]
            },
            {
                "question": "How often is Sales Tax audit conducted?",
                "answer": "Ordinarily, Sales Tax audit is conducted once a year.",
                "keywords": ["audit", "how often", "frequency", "yearly"]
            },
            {
                "question": "What is the appellate procedure?",
                "answer": "The appellate procedure is same as in Federal Fiscal Laws. Appellate Forums: (i) Commissioner (Appeals), (ii) Appellate Tribunal, and (iii) High Court for questions of law. Administrative Forums: (i) Alternate Dispute Resolution Committee, and (ii) Tax Ombudsman.",
                "keywords": ["appeal", "appellate", "dispute", "complaint", "tribunal"]
            },
            {
                "question": "Will existing registration be valid under new Act?",
                "answer": "There is no need to re-register as all existing registered persons shall be deemed to be registered under the Balochistan Sales Tax Act, 2015 and registered person's registration shall be NTN with prefix 'B'.",
                "keywords": ["existing registration", "re-register", "valid", "NTN"]
            },
            {
                "question": "How can I download registration form BST-01?",
                "answer": "You can download the Registration Form BST-01 from our system. This form is required for Sales Tax on Services registration and must be submitted within 30 days of starting taxable services.",
                "keywords": ["download", "registration form", "BST-01", "form", "download form"]
            },
            {
                "question": "Where can I get the payment challan BST-04?",
                "answer": "The Payment Challan BST-04 is available for download. Use this form to pay Sales Tax on Services at authorized National Bank branches with head of account 'B-02387 - Balochistan Sales Tax on Services'.",
                "keywords": ["payment challan", "BST-04", "challan", "download", "payment form"]
            },
            {
                "question": "How to download return form BST-03?",
                "answer": "The Monthly Return Form BST-03 is available for download. This form must be filed electronically through BRA portal by the 15th of the following month.",
                "keywords": ["return form", "BST-03", "monthly return", "download", "filing form"]
            },
            {
                "question": "Who is the Chairperson of BRA?",
                "answer": "Mr. Abdullah Khan is the current Chairperson of Balochistan Revenue Authority. He was transferred and posted as Chairperson BRA w.e.f. 19th June, 2025. You can contact him at chairperson@bra.gob.pk or 081-2452073, Ext: 0.",
                "keywords": ["chairperson", "chairman", "abdullah khan", "head", "chief"]
            },
            {
                "question": "Who is the IT Member at BRA?",
                "answer": "Mr. Hassan Mengal is the Member IT at Balochistan Revenue Authority. You can contact him at member.it@bra.gob.pk or 081-2452073, Ext: 0.",
                "keywords": ["IT member", "member IT", "hassan mengal", "information technology", "tech"]
            },
            {
                "question": "Who handles HR matters at BRA?",
                "answer": "Mr. Ehtisham-ul-Haq is the Member HR at Balochistan Revenue Authority. You can contact him at member.hr@bra.gob.pk or 081-2452073, Ext: 109.",
                "keywords": ["HR member", "member HR", "human resources", "ehtisham", "staff matters"]
            },
            {
                "question": "Who is responsible for audit at BRA?",
                "answer": "Mr. Syed Sikandar Shah is the Member Audit at Balochistan Revenue Authority. You can contact him at smember.audit@bra.gob.pk or 081-2452073, Ext: 116.",
                "keywords": ["audit member", "member audit", "sikandar shah", "auditing", "audit matters"]
            },
            {
                "question": "What is the Balochistan Infrastructure Cess Act?",
                "answer": "The Balochistan Infrastructure Cess Act, 2021 (Amended 2024) establishes a system to collect a cess (tax) on goods transported in, out, or through Balochistan by road, rail, air, sea, or pipelines. The revenue is used exclusively for the development and maintenance of provincial infrastructure like roads, bridges, hospitals, schools, and public facilities.",
                "keywords": ["infrastructure cess", "cess act", "balochistan infrastructure", "infrastructure tax", "cess 2021", "amended 2024"]
            },
            {
                "question": "What are the Infrastructure Cess rates?",
                "answer": "Infrastructure Cess rates range from 1.10% to 1.15% of the value of goods plus an additional fee per kilometer based on distance. Gold is taxed at 0.125% of its value. Goods leaving Balochistan for export are taxed at 0%. The cess is based on value, weight, and distance of transported goods.",
                "keywords": ["cess rates", "infrastructure cess rates", "1.10%", "1.15%", "gold tax", "export tax"]
            },
            {
                "question": "What goods are covered under Infrastructure Cess?",
                "answer": "The Infrastructure Cess Act covers all goods manufactured, consumed, imported, exported, or transported through Balochistan by road, rail, air, sea, or pipelines. It applies to the entire province of Balochistan.",
                "keywords": ["covered goods", "cess goods", "transported goods", "manufactured goods", "imported goods"]
            },
            {
                "question": "How is Infrastructure Cess collected?",
                "answer": "Infrastructure Cess is collected like customs duty or sales tax. Payments are deposited in the 'Infrastructure Cess' account. The Balochistan Revenue Authority (BRA) manages collection and enforcement. BRA can integrate its systems with FBR, PSW, and other government bodies for better monitoring.",
                "keywords": ["cess collection", "how to pay cess", "infrastructure cess payment", "BRA collection"]
            },
            {
                "question": "What are the penalties for Infrastructure Cess evasion?",
                "answer": "Non-payment or evasion of Infrastructure Cess can lead to penalties equal to the evaded amount. BRA has the authority to seize goods or vehicles involved in cess evasion. Strict rules govern auctions of confiscated goods and provisional release of seized vehicles.",
                "keywords": ["cess penalties", "cess evasion", "seizure", "confiscated goods", "infrastructure cess penalty"]
            },
            {
                "question": "How to appeal Infrastructure Cess decisions?",
                "answer": "Appeals against Infrastructure Cess decisions must be filed within 30 days of an order. Refund claims must be submitted within 3 months of payment. The appeals process follows standard BRA procedures.",
                "keywords": ["cess appeal", "infrastructure cess appeal", "refund claim", "30 days", "3 months"]
            },
            {
                "question": "What are Infrastructure Cess funds used for?",
                "answer": "Infrastructure Cess funds are used exclusively for the development and maintenance of provincial infrastructure, including roads, bridges, public parks, hospitals, schools, streetlights, and safety services. The funds are strictly dedicated to infrastructure projects.",
                "keywords": ["cess funds", "infrastructure development", "roads", "bridges", "hospitals", "schools", "fund usage"]
            }
        ]
        
        # Response patterns for common greetings and queries
        self.response_patterns = [
            {
                "patterns": [r"hello|hi|hey|salam|assalam", r"good morning|good afternoon|good evening"],
                "responses": [
                    "Assalam-o-Alaikum! Welcome to Balochistan Revenue Authority. How can I assist you today?",
                    "Hello! I'm here to help you with BRA services and tax information. What would you like to know?",
                    "Hi there! I can help you with questions about BRA services, registration, tax rates, and more. How may I assist you?"
                ]
            },
            {
                "patterns": [r"thank you|thanks|shukriya"],
                "responses": [
                    "You're welcome! Feel free to ask if you have any other questions about BRA services.",
                    "Happy to help! If you need more information, I'm here to assist.",
                    "My pleasure! Don't hesitate to reach out if you need more help with BRA services."
                ]
            },
            {
                "patterns": [r"bye|goodbye|khuda hafiz"],
                "responses": [
                    "Goodbye! Have a great day and thank you for using BRA services.",
                    "Khuda Hafiz! Feel free to return anytime for BRA assistance.",
                    "Take care! Remember, BRA is here to serve you."
                ]
            },
            {
                "patterns": [r"^(?!.*(?:commissioner|assistant commissioner|management|team|staff)).*(?:help|assist|support|guide|madad)"],
                "responses": [
                    "I'm here to help! I can assist you with:\n• BRA registration and services\n• Tax payments and returns\n• Download forms and documents\n• Contact information\n• Infrastructure Cess queries\n\nWhat specific help do you need?",
                    "How can I help you today? I can provide information about BRA services, forms, payments, and more. Just ask me anything!",
                    "I'm your BRA assistant! I can help with registration, tax payments, form downloads, and general BRA information. What would you like to know?"
                ]
            },
            {
                "patterns": [r"what can you do|capabilities|features|services you provide"],
                "responses": [
                    "I can help you with:\n• BRA introduction and information\n• Tax rates and regulations\n• Registration and enrollment\n• Payment methods\n• Form downloads\n• Contact details\n• Infrastructure Cess information\n• Management details\n\nWhat would you like to know?",
                    "I'm your BRA assistant! I can provide information about BRA services, help with tax queries, provide download links for forms, and answer questions about BRA operations. What do you need help with?",
                    "I can assist you with BRA services, tax information, registration help, form downloads, and general BRA queries. Just ask me anything about BRA!"
                ]
            },
            {
                "patterns": [r"who are you|what are you|introduce yourself"],
                "responses": [
                    "I'm the BRA (Balochistan Revenue Authority) chatbot! I'm here to help you with BRA services, tax information, registration, payments, and any questions you have about BRA operations.",
                    "Hello! I'm your BRA assistant, designed to help you with Balochistan Revenue Authority services and information. How can I assist you today?",
                    "I'm the BRA chatbot, your digital assistant for all Balochistan Revenue Authority related queries. I can help with services, forms, payments, and more!"
                ]
            },
            {
                "patterns": [r"nice|good|great|awesome|excellent|wonderful|amazing|fantastic|perfect|brilliant"],
                "responses": [
                    "Thank you! I'm glad I could help. Is there anything else about BRA services you'd like to know?",
                    "I appreciate your feedback! I'm here to assist you with any BRA-related questions you might have.",
                    "That's wonderful to hear! Feel free to ask me anything else about BRA services, forms, or procedures.",
                    "Thank you for the positive feedback! I'm always here to help with BRA information and services."
                ]
            },
            {
                "patterns": [r"cool|sweet|lovely|beautiful|amazing|impressive|outstanding"],
                "responses": [
                    "Thank you! I'm designed to provide the best possible assistance with BRA services. What else can I help you with?",
                    "I'm glad you find it helpful! I'm here to make BRA services more accessible for everyone. Any other questions?",
                    "That's kind of you to say! I'm always ready to help with BRA-related queries. What would you like to know next?"
                ]
            },
            {
                "patterns": [r"ok|okay|alright|sure|fine|good|yes|yeah|yep"],
                "responses": [
                    "Great! I'm here whenever you need help with BRA services. What would you like to know?",
                    "Perfect! Feel free to ask me anything about BRA operations, forms, or services.",
                    "Excellent! I'm ready to assist you with any BRA-related questions you have."
                ]
            },
            {
                "patterns": [r"no problem|no worries|it's fine|don't worry|not a problem"],
                "responses": [
                    "You're very welcome! I'm here to help make BRA services easier for you. Anything else I can assist with?",
                    "My pleasure! I'm designed to help with all BRA-related queries. Feel free to ask anytime!",
                    "Happy to help! I'm always available to assist with BRA services and information."
                ]
            },
            {
                "patterns": [r"you're welcome|welcome|pleasure|my pleasure"],
                "responses": [
                    "Thank you! I'm here to provide the best possible assistance with BRA services. What else can I help you with?",
                    "I appreciate that! I'm always ready to help with BRA-related questions. Anything specific you'd like to know?",
                    "That's very kind! I'm designed to make BRA services more accessible. Feel free to ask me anything!"
                ]
            },
            {
                "patterns": [r"how are you|how do you do|how's it going|what's up"],
                "responses": [
                    "I'm doing great, thank you for asking! I'm here and ready to help you with BRA services. How can I assist you today?",
                    "I'm functioning perfectly and ready to help! What BRA-related questions do you have for me?",
                    "I'm doing well, thank you! I'm here to assist you with all BRA services and information. What do you need help with?"
                ]
            },
            {
                "patterns": [r"good morning|good afternoon|good evening|good night"],
                "responses": [
                    "Good morning! Welcome to BRA services. How can I assist you today?",
                    "Good afternoon! I'm here to help with BRA information and services. What would you like to know?",
                    "Good evening! I'm ready to assist you with BRA-related queries. How may I help you?",
                    "Good night! I'm here 24/7 to help with BRA services. What can I do for you?"
                ]
            },
            {
                "patterns": [r"please|kindly|could you|would you|can you help"],
                "responses": [
                    "Of course! I'm here to help with BRA services. What specific assistance do you need?",
                    "Absolutely! I'm designed to assist with BRA-related queries. What would you like to know?",
                    "Certainly! I'm ready to help with BRA services and information. What do you need assistance with?"
                ]
            },
            {
                "patterns": [r"sorry|apologize|excuse me|pardon|my bad"],
                "responses": [
                    "No need to apologize! I'm here to help with BRA services. What can I assist you with?",
                    "No worries at all! I'm designed to be patient and helpful. What BRA-related questions do you have?",
                    "Don't worry about it! I'm here to make BRA services easier for you. How can I help?"
                ]
            },
            {
                "patterns": [r"i don't know|not sure|confused|unclear|don't understand"],
                "responses": [
                    "No problem! I'm here to help clarify BRA services and procedures. What specific information do you need?",
                    "That's perfectly fine! I'm designed to explain BRA services in simple terms. What would you like to know?",
                    "I understand! Let me help you with BRA information. What specific question do you have?"
                ]
            },
            {
                "patterns": [r"maybe|perhaps|might be|could be|possibly"],
                "responses": [
                    "I'm here to help you find the right information about BRA services. What specific question do you have?",
                    "Let me assist you with BRA information. What would you like to know about our services?",
                    "I can help clarify BRA procedures for you. What specific assistance do you need?"
                ]
            }
        ]

class BRAChatbot:
    """Main chatbot class that processes queries and generates responses"""
    
    def __init__(self):
        self.kb = BRAKnowledgeBase()
        self.conversation_history = []
    
    def preprocess_query(self, query):
        """Clean and normalize user query"""
        query = query.lower().strip()
        # Remove special characters but keep spaces
        query = re.sub(r'[^\w\s]', ' ', query)
        # Remove extra spaces
        query = re.sub(r'\s+', ' ', query)
        return query
    
    def fuzzy_match_keywords(self, query, keywords):
        """Check if query matches keywords with fuzzy matching for typos"""
        query_words = query.split()
        
        for keyword in keywords:
            keyword_words = keyword.lower().split()
            
            # Check exact match first
            if keyword.lower() in query:
                return True
            
            # Check if all keyword words are present (allows for word order variations)
            if all(any(self.similar_word(kw, qw) for qw in query_words) for kw in keyword_words):
                return True
        
        return False
    
    def similar_word(self, word1, word2):
        """Check if two words are similar (handles common typos)"""
        # Exact match
        if word1 == word2:
            return True
        
        # Handle common variations and typos
        variations = {
            'chairperson': ['chairman', 'chair', 'chairmen', 'chairpersn', 'chairperson'],
            'member': ['memeber', 'memmber', 'memmer', 'member'],
            'it': ['IT', 'information', 'tech', 'technology'],
            'hr': ['HR', 'human', 'resources'],
            'audit': ['auditing', 'auditor'],
            'commissioner': ['commisioner', 'comissioner'],
            'management': ['managment', 'team', 'staff'],
            'download': ['donwload', 'dwonload', 'dwnload'],
            'registration': ['registeration', 'registraton'],
            'challan': ['chalan', 'challaan', 'challen']
        }
        
        # Check if either word is in the variations of the other
        for key, variants in variations.items():
            if (word1 == key and word2 in variants) or (word2 == key and word1 in variants):
                return True
            if word1 in variants and word2 in variants:
                return True
        
        # Simple character distance check for typos (Levenshtein distance approximation)
        if len(word1) > 3 and len(word2) > 3:
            if self.simple_distance(word1, word2) <= 2:
                return True
        
        return False
    
    def simple_distance(self, s1, s2):
        """Calculate simple edit distance between two strings"""
        if len(s1) < len(s2):
            return self.simple_distance(s2, s1)
        
        if len(s2) == 0:
            return len(s1)
        
        previous_row = list(range(len(s2) + 1))
        for i, c1 in enumerate(s1):
            current_row = [i + 1]
            for j, c2 in enumerate(s2):
                insertions = previous_row[j + 1] + 1
                deletions = current_row[j] + 1
                substitutions = previous_row[j] + (c1 != c2)
                current_row.append(min(insertions, deletions, substitutions))
            previous_row = current_row
        
        return previous_row[-1]
    
    def find_best_faq_match(self, query):
        """Find the best matching FAQ based on keywords"""
        query_words = set(query.split())
        best_match = None
        best_score = 0
        
        for faq in self.kb.faqs:
            # Calculate keyword match score
            keyword_matches = sum(1 for keyword in faq["keywords"] if keyword in query)
            
            # Calculate word overlap score
            faq_words = set(" ".join(faq["keywords"]).split())
            word_overlap = len(query_words.intersection(faq_words))
            
            total_score = keyword_matches * 2 + word_overlap
            
            if total_score > best_score:
                best_score = total_score
                best_match = faq
        
        return best_match if best_score > 0 else None
    
    def check_response_patterns(self, query):
        """Check if query matches common response patterns"""
        for pattern_group in self.kb.response_patterns:
            for pattern in pattern_group["patterns"]:
                if re.search(pattern, query, re.IGNORECASE):
                    return random.choice(pattern_group["responses"])
        return None

    def handle_bra_introduction_query(self, processed_query):
        """Handle BRA introduction and general information queries"""
        query_lower = processed_query.lower()
        
        # BRA introduction keywords
        intro_keywords = [
            "introduction", "about bra", "what is bra", "bra info", 
            "bra details", "tell me about bra", "bra overview"
        ]
        
        # BRA services keywords  
        services_keywords = [
            "bra services", "services", "e-services", "what services", 
            "bra offers", "available services", "service list"
        ]
        
        # Check for BRA introduction queries
        if any(self.fuzzy_match_keywords(query_lower, keywords) for keywords in [intro_keywords]):
            try:
                intro = self.kb.bra_info['introduction']
                contact = self.kb.bra_info['contact']
                response_text = f"## 🏛️ **{intro['title']}**\n\n### **📋 Introduction**\n{intro['description']}\n\n### **🎯 Our Vision**\n{intro['vision']}\n\n### **🚀 Our Mission**\n{intro['mission']}\n\n### **💎 Our Values**\n{intro['values']}\n\n### **🌟 Our Goal**\n{intro['goal']}\n\n### **📞 Contact Information**\n- **Phone:** {contact['phone']}\n- **Email:** {contact['email']}\n- **Address:** {contact['address']}\n- **Hours:** {contact['hours']}\n\nFor more information about BRA services and operations, please feel free to ask!"
                return {
                    "response": response_text,
                    "type": "info"
                }
            except Exception as e:
                print(f"Error in BRA introduction: {e}")
                return {
                    "response": "I'm sorry, I encountered an error while retrieving BRA information. Please try again or contact BRA directly.",
                    "type": "error"
                }
        
        # Check for BRA services queries
        if any(self.fuzzy_match_keywords(query_lower, keywords) for keywords in [services_keywords]):
            try:
                services = self.kb.bra_info['services']
                contact = self.kb.bra_info['contact']
                services_list = []
                
                for service_key, service_info in services.items():
                    service_text = f"**{service_info['name']}**\n{service_info['description']}"
                    if 'sub_services' in service_info:
                        service_text += "\n- " + "\n- ".join(service_info['sub_services'])
                    services_list.append(service_text)
                
                response_text = f"## 🛠️ **BRA e-Services**\n\n### **📋 Available Services**\n{chr(10).join(services_list)}\n\n### **🌐 Access Our Services**\nVisit our official website at [ebra.com.pk](https://ebra.com.pk) to access all e-Services.\n\n### **📞 Need Help?**\n- **Phone:** {contact['phone']}\n- **Email:** {contact['email']}\n- **Hours:** {contact['hours']}\n\nOur professional team is ready to assist you with any service-related queries!"
                return {
                    "response": response_text,
                    "type": "info"
                }
            except Exception as e:
                print(f"Error in BRA services: {e}")
                return {
                    "response": "I'm sorry, I encountered an error while retrieving BRA services information. Please try again or contact BRA directly.",
                    "type": "error"
                }
        
        return None
    
    def handle_management_query(self, processed_query):
        """Handle queries about BRA management and team members"""
        management = self.kb.bra_info["management"]
        
        # Check for management structure queries first (highest priority)
        structure_keywords = ["management structure", "bra management", "management team", "team structure", "organization structure", "bra team", "management overview", "team overview"]
        if self.fuzzy_match_keywords(processed_query, structure_keywords):
            # General management information
            team_info = f"👨‍💼 **BRA Management Team**\n\n"
            team_info += f"**Chairperson:** {management['chairperson']['name']}\n"
            team_info += f"**Members:**\n"
            for member in management["members"]:
                team_info += f"• {member['name']} - {member['position']}\n"
            team_info += f"\n**Commissioners:**\n"
            for comm in management["commissioners"]:
                team_info += f"• {comm['name']} - {comm['position']}\n"
            team_info += f"\n**Assistant Commissioners:** {len(management['assistant_commissioners'])} members\n"
            team_info += f"\n💡 **Ask specifically about:**\n• Chairperson details\n• IT Member information\n• HR Member details\n• Audit Member info\n• Commissioners list\n• Assistant Commissioners\n\nFor complete team details, visit: https://ebra.com.pk/management"
            
            return {
                "response": team_info,
                "type": "management",
                "confidence": "high"
            }
        
        # Check for specific management queries using fuzzy matching
        chairperson_keywords = ["chairperson", "chairman", "head", "chief", "chair person", "chair man"]
        if self.fuzzy_match_keywords(processed_query, chairperson_keywords):
            chair = management["chairperson"]
            return {
                "response": f"👨‍💼 **BRA Chairperson**\n\n**{chair['name']}** - {chair['position']}\n\n📞 Phone: {chair['phone']}\n📧 Email: {chair['email']}\n\n💼 Mr. Abdullah Khan was transferred and posted as Chairperson BRA w.e.f. 19th June, 2025.\n\nFor more information, visit: https://ebra.com.pk/management",
                "type": "management",
                "confidence": "high"
            }
        
        elif self.fuzzy_match_keywords(processed_query, ["IT member", "member IT", "hassan", "information technology", "tech", "I T member", "it memeber", "IT memmber"]):
            it_member = next((m for m in management["members"] if "IT" in m["position"]), None)
            if it_member:
                return {
                    "response": f"💻 **BRA IT Member**\n\n**{it_member['name']}** - {it_member['position']}\n\n📞 Phone: {it_member['phone']}\n📧 Email: {it_member['email']}\n\nMr. Hassan Mengal handles all IT and technology-related matters at BRA.\n\nFor more information, visit: https://ebra.com.pk/management",
                    "type": "management",
                    "confidence": "high"
                }
        
        elif self.fuzzy_match_keywords(processed_query, ["HR member", "member HR", "human resources", "ehtisham", "H R member", "hr memeber", "HR memmber"]) and not any(word in processed_query for word in ["structure", "overview", "team", "management"]):
            hr_member = next((m for m in management["members"] if "HR" in m["position"]), None)
            if hr_member:
                return {
                    "response": f"👥 **BRA HR Member**\n\n**{hr_member['name']}** - {hr_member['position']}\n\n📞 Phone: {hr_member['phone']}\n📧 Email: {hr_member['email']}\n\nMr. Ehtisham-ul-Haq handles all human resources and staff matters at BRA.\n\nFor more information, visit: https://ebra.com.pk/management",
                    "type": "management",
                    "confidence": "high"
                }
        
        elif self.fuzzy_match_keywords(processed_query, ["audit member", "member audit", "sikandar", "auditing", "audit memeber", "member auditt"]):
            audit_member = next((m for m in management["members"] if "Audit" in m["position"]), None)
            if audit_member:
                return {
                    "response": f"📊 **BRA Audit Member**\n\n**{audit_member['name']}** - {audit_member['position']}\n\n📞 Phone: {audit_member['phone']}\n📧 Email: {audit_member['email']}\n\nMr. Syed Sikandar Shah is responsible for all audit-related matters at BRA.\n\nFor more information, visit: https://ebra.com.pk/management",
                    "type": "management",
                    "confidence": "high"
                }
        
        elif self.fuzzy_match_keywords(processed_query, ["it team", "bra it team", "it department", "it staff", "technology team", "it members", "it personnel", "it support", "technical support", "it help", "system support", "it assistance", "tech support"]):
            # Handle IT team queries
            it_team_info = f"💻 **BRA IT Team**\n\n"
            for it_member in management["it_team"]:
                it_team_info += f"**{it_member['name']}** - {it_member['position']}\n"
                it_team_info += f"📞 Phone: {it_member['phone']}\n"
                it_team_info += f"📧 Email: {it_member['email']}\n"
                it_team_info += f"🔧 Role: {it_member['role']}\n\n"
            it_team_info += f"💡 **IT Team Responsibilities:**\n• System development and maintenance\n• IT infrastructure management\n• Technical support and assistance\n• Digital services development\n\nFor complete information, visit: https://ebra.com.pk/management"
            
            return {
                "response": it_team_info,
                "type": "management",
                "confidence": "high"
            }
        
        elif self.fuzzy_match_keywords(processed_query, ["sadaqat khan", "full stack developer", "developer", "programmer", "software developer", "it developer"]):
            # Handle Sadaqat Khan queries
            sadaqat = next((m for m in management["it_team"] if "Sadaqat" in m["name"]), None)
            if sadaqat:
                return {
                    "response": f"👨‍💻 **BRA Full Stack Developer**\n\n**{sadaqat['name']}** - {sadaqat['position']}\n\n📞 Phone: {sadaqat['phone']}\n📧 Email: {sadaqat['email']}\n🔧 Role: {sadaqat['role']}\n\nMr. Sadaqat Khan is currently hired by PRAL but works with the BRA IT Team on system development and technical projects.\n\nFor more information, visit: https://ebra.com.pk/management",
                    "type": "management",
                    "confidence": "high"
                }
        
        elif self.fuzzy_match_keywords(processed_query, ["commissioners", "commissioner", "commisioner", "commisioners"]):
            # Handle commissioner queries
            commissioners_info = f"👨‍💼 **BRA Commissioners**\n\n"
            for comm in management["commissioners"]:
                commissioners_info += f"**{comm['name']}** - {comm['position']}\n"
                commissioners_info += f"📞 Phone: {comm['phone']}\n"
                commissioners_info += f"📧 Email: {comm['email']}\n\n"
            commissioners_info += f"💡 **Ask specifically about:**\n• Individual commissioner details\n• Assistant Commissioners\n• Management team overview\n\nFor complete information, visit: https://ebra.com.pk/management"
            
            return {
                "response": commissioners_info,
                "type": "management",
                "confidence": "high"
            }
        
        elif self.fuzzy_match_keywords(processed_query, ["assistant commissioners", "assistant commissioner", "ac", "assistant commisioner", "assistant commisioners", "asssitant comissioners", "asssitant comissioner", "assistant comissioners", "assistant comissioner", "asssitant commissioners", "asssitant commissioner", "list assistant", "assistant comm", "ass comm", "assistant comissionners", "assistant comissionner", "asssitant comissionners", "asssitant comissionner", "assistant commisionners", "assistant commisionner", "asssitant commisionners", "asssitant commisionner", "assistant comissioners", "assistant comissioner", "asssitant comissioners", "asssitant comissioner"]):
            # Handle assistant commissioner queries
            ac_info = f"👨‍💼 **BRA Assistant Commissioners**\n\n"
            for ac in management["assistant_commissioners"]:
                ac_info += f"**{ac['name']}** - {ac['position']}\n"
                ac_info += f"📞 Phone: {ac['phone']}\n"
                ac_info += f"📧 Email: {ac['email']}\n\n"
            ac_info += f"💡 **Ask specifically about:**\n• Individual assistant commissioner details\n• Commissioners\n• Management team overview\n\nFor complete information, visit: https://ebra.com.pk/management"
            
            return {
                "response": ac_info,
                "type": "management",
                "confidence": "high"
            }
        
        
        return {
            "response": "I can help you with information about BRA management team. You can ask about:\n• Chairperson details\n• IT Member\n• HR Member\n• Audit Member\n• Management team overview\n\nFor complete information, visit: https://ebra.com.pk/management",
            "type": "management",
            "confidence": "medium"
        }

    def handle_infrastructure_cess_query(self, processed_query):
        """Handle queries about Infrastructure Cess Act"""
        cess_info = self.kb.bra_info["infrastructure_cess"]
        
        # Check for specific cess queries
        if self.fuzzy_match_keywords(processed_query, ["cess rates", "rates", "percentage", "how much", "tax rate"]):
            rates = cess_info["rates"]
            return {
                "response": f"🏗️ **Infrastructure Cess Rates:**\n\n📊 **General Goods:** {rates['general']}\n💰 **Gold:** {rates['gold']}\n📦 **Exports:** {rates['exports']}\n\n📏 **Calculation:** Cess is based on value, weight, and distance of transported goods.\n\n⚖️ **Coverage:** All goods transported in, out, or through Balochistan by road, rail, air, sea, or pipelines.\n\n📋 **Act:** {cess_info['act_name']}",
                "type": "infrastructure_cess",
                "confidence": "high"
            }
        
        elif self.fuzzy_match_keywords(processed_query, ["fund usage", "funds used", "what for", "infrastructure development", "where money goes"]):
            usage_list = "\n".join([f"• {usage}" for usage in cess_info["fund_usage"]])
            return {
                "response": f"🏗️ **Infrastructure Cess Fund Usage:**\n\n💰 **Funds are exclusively used for:**\n{usage_list}\n\n🎯 **Purpose:** {cess_info['purpose']}\n\n📋 **Act:** {cess_info['act_name']}\n\n✅ **Strict Rule:** Funds are dedicated solely to infrastructure projects and maintenance.",
                "type": "infrastructure_cess",
                "confidence": "high"
            }
        
        elif self.fuzzy_match_keywords(processed_query, ["penalties", "penalty", "evasion", "seizure", "confiscation"]):
            return {
                "response": f"⚠️ **Infrastructure Cess Penalties:**\n\n💸 **Penalty Amount:** {cess_info['penalties']}\n\n🚫 **BRA Authority:**\n• Seize goods or vehicles involved in cess evasion\n• Auction confiscated goods under strict rules\n• Provisional release of seized vehicles\n\n⏰ **Appeals & Refunds:**\n• Appeals: {cess_info['appeals'].split(',')[0]}\n• Refund claims: {cess_info['appeals'].split(',')[1]}\n\n📋 **Act:** {cess_info['act_name']}",
                "type": "infrastructure_cess",
                "confidence": "high"
            }
        
        elif self.fuzzy_match_keywords(processed_query, ["covered goods", "what goods", "applicable goods", "which goods"]):
            return {
                "response": f"📦 **Goods Covered by Infrastructure Cess:**\n\n✅ **All goods that are:**\n• Manufactured in Balochistan\n• Consumed in Balochistan\n• Imported through Balochistan\n• Exported from Balochistan\n• Transported through Balochistan\n\n🚛 **Transport Methods:**\n• Road transportation\n• Rail transportation\n• Air transportation\n• Sea transportation\n• Pipeline transportation\n\n🗺️ **Coverage:** {cess_info['coverage']}\n\n📋 **Act:** {cess_info['act_name']}",
                "type": "infrastructure_cess",
                "confidence": "high"
            }
        
        elif self.fuzzy_match_keywords(processed_query, ["how to pay", "collection", "payment process", "how collected"]):
            return {
                "response": f"💳 **Infrastructure Cess Collection:**\n\n📋 **Collection Method:**\n• Collected like customs duty or sales tax\n• Payments deposited in 'Infrastructure Cess' account\n• Managed by Balochistan Revenue Authority (BRA)\n\n🔗 **System Integration:**\n• BRA systems integrated with FBR, PSW\n• Connected with other government bodies\n• Enhanced monitoring and enforcement\n\n🏢 **Private Engagement:**\n• Private companies may assist in collection\n• Professional enforcement services\n\n📋 **Act:** {cess_info['act_name']}",
                "type": "infrastructure_cess",
                "confidence": "high"
            }
        
        else:
            # General Infrastructure Cess information
            return {
                "response": f"🏗️ **{cess_info['act_name']}**\n\n🎯 **Purpose:** {cess_info['purpose']}\n\n📊 **Key Rates:**\n• General: {cess_info['rates']['general']}\n• Gold: {cess_info['rates']['gold']}\n• Exports: {cess_info['rates']['exports']}\n\n🗺️ **Coverage:** {cess_info['coverage']}\n\n💡 **Ask specifically about:**\n• Cess rates and calculation\n• Fund usage and infrastructure projects\n• Penalties and enforcement\n• Covered goods and transportation\n• Payment and collection process",
                "type": "infrastructure_cess",
                "confidence": "high"
            }

    def handle_document_request(self, processed_query):
        """Handle requests for document downloads"""
        documents = self.kb.bra_info["documents"]
        
        # Check for specific document requests
        if any(word in processed_query for word in ["BST-01", "registration form", "registration"]):
            doc = documents["BST-01"]
            return {
                "response": f"📄 **{doc['name']}** is ready for download!\n\n✅ {doc['description']}\n\n⬇️ [📥 DOWNLOAD REGISTRATION FORM BST-01](/download/BST-01)\n\n📋 **What you need to do:**\n• Fill the form in BLOCK LETTERS\n• Submit within 30 days of starting taxable services\n• Required documents: CNIC, NTN Certificate, Business documents\n• Submit online at BRA portal or visit our office\n\n📞 Need help? Call: 081-2452073",
                "type": "document",
                "document": "BST-01",
                "download_url": "/download/BST-01",
                "confidence": "high"
            }
        
        elif any(word in processed_query for word in ["BST-04", "payment challan", "challan", "payment form", "give me challan"]):
            doc = documents["BST-04"]
            return {
                "response": f"💳 **{doc['name']}** is ready for download!\n\n✅ {doc['description']}\n\n⬇️ [📥 DOWNLOAD PAYMENT CHALLAN BST-04](/download/BST-04)\n\n💰 **Payment Instructions:**\n• Use ONLY authorized National Bank branches\n• Head of account: B-02387 - Balochistan Sales Tax on Services\n• Fill in BLOCK LETTERS\n• Keep depositor copy for your records\n• Payment must be made by due date to avoid penalties\n\n🏦 For bank branch list, visit: ebra.com.pk",
                "type": "document",
                "document": "BST-04",
                "download_url": "/download/BST-04",
                "confidence": "high"
            }
        
        elif any(word in processed_query for word in ["BST-03", "return form", "monthly return", "filing form"]):
            doc = documents["BST-03"]
            return {
                "response": f"📊 **{doc['name']}** is ready for download!\n\n✅ {doc['description']}\n\n⬇️ [📥 DOWNLOAD RETURN FORM BST-03](/download/BST-03)\n\n📅 **Filing Requirements:**\n• File electronically through BRA portal\n• Submit by 15th of following month\n• Keep supporting documents for 5 years\n• Late filing penalty: Rs. 1,000 per day\n• Include all taxable services and input tax\n\n💡 File online at: ebra.com.pk",
                "type": "document",
                "document": "BST-03",
                "download_url": "/download/BST-03",
                "confidence": "high"
            }
        
        else:
            # Show comprehensive download categories from BRA website
            download_categories = self.kb.bra_info["download_categories"]
            
            # Check if asking for general downloads/forms
            if any(word in processed_query for word in ["downloads", "all forms", "categories", "what can i download"]):
                category_list = []
                for key, category in download_categories.items():
                    category_list.append(f"📂 **{category['name']}** - {category['description']}")
                
                response = f"📋 **BRA Download Categories** (from ebra.com.pk/downloads):\n\n" + "\n\n".join(category_list)
                response += f"\n\n📄 **Quick Download Forms:**\n"
                for key, doc in documents.items():
                    response += f"• [📥 DOWNLOAD {doc['name'].upper()}](/download/{key})\n"
                response += f"\n🌐 **For complete downloads, visit:** https://ebra.com.pk/downloads"
                
                return {
                    "response": response,
                    "type": "downloads",
                    "confidence": "high"
                }
            else:
                # General document information with download links
                doc_list = []
                for key, doc in documents.items():
                    doc_list.append(f"📄 **{doc['name']}** - {doc['description']}\n⬇️ [📥 DOWNLOAD {doc['name'].upper()}](/download/{key})")
                
                return {
                    "response": f"📋 **Available BRA Forms & Documents:**\n\n" + "\n\n".join(doc_list) + f"\n\n💡 **More Downloads Available:**\n🌐 Visit https://ebra.com.pk/downloads for:\n• Acts & Legislation\n• Rules & Regulations\n• Tax Schedules\n• NBP Authorized Branches\n• Circulars & Notifications\n• And much more!",
                    "type": "documents",
                    "confidence": "high"
                }

    def generate_response(self, user_query):
        """Generate appropriate response for user query"""
        processed_query = self.preprocess_query(user_query)
        
        # Store conversation
        self.conversation_history.append({
            "user": user_query,
            "timestamp": datetime.now().isoformat()
        })
        
        # Check for common response patterns first
        pattern_response = self.check_response_patterns(processed_query)
        if pattern_response:
            return {
                "response": pattern_response,
                "type": "greeting",
                "confidence": "high"
            }
        
        # Handle management and team queries FIRST (HIGHEST PRIORITY)
        management_keywords = ["management", "team", "staff", "who is", "chairperson", "chairman", "member", "commissioner", "managment", "chair person", "chair man", "memeber", "commisioner", "assistant commissioner", "assistant commissioners", "ac", "commisioners", "commissioners", "assistant commisioner", "assistant commisioners", "assistant comissionners", "assistant comissionner", "asssitant comissionners", "asssitant comissionner", "assistant commisionners", "assistant commisionner", "asssitant commisionners", "asssitant commisionner", "assistant comissioners", "assistant comissioner", "asssitant comissioners", "asssitant comissioner", "it team", "bra it team", "it department", "it staff", "technology team", "it members", "it personnel", "it support", "technical support", "it help", "system support", "it assistance", "tech support", "sadaqat khan", "full stack developer", "developer", "programmer", "software developer", "it developer"]
        if self.fuzzy_match_keywords(processed_query, management_keywords):
            return self.handle_management_query(processed_query)
        
        # Handle document download requests (with more specific keywords)
        download_keywords = ["download", "form", "BST-01", "BST-03", "BST-04", "registration form", "challan form", "challan", "return form", "payment form", "donwload", "registeration", "chalan"]
        # Only match download if it's specifically about forms/documents, not general "give me"
        if any(word in processed_query for word in ["download", "form", "BST-01", "BST-03", "BST-04", "challan", "chalan"]) or \
           (any(word in processed_query for word in ["give me", "get me"]) and any(word in processed_query for word in ["form", "document", "challan", "registration", "return", "payment"])):
            return self.handle_document_request(processed_query)
        
        # Look for FAQ matches
        faq_match = self.find_best_faq_match(processed_query)
        if faq_match:
            return {
                "response": faq_match["answer"],
                "type": "faq",
                "question": faq_match["question"],
                "confidence": "high"
            }
        
        # Handle specific information requests
        if any(word in processed_query for word in ["contact", "phone", "email", "address"]):
            contact = self.kb.bra_info["contact"]
            return {
                "response": f"Here's how you can contact BRA:\n📞 Phone: {contact['phone']}\n📧 Email: {contact['email']}\n📍 Address: {contact['address']}\n🕒 Hours: {contact['hours']}",
                "type": "contact",
                "confidence": "high"
            }
        
        if any(word in processed_query for word in ["services", "e-services", "online"]):
            services = ", ".join(self.kb.bra_info["services"])
            return {
                "response": f"BRA provides the following e-services: {services}. You can access these services through the BRA website.",
                "type": "services",
                "confidence": "high"
            }
        
        
        # Handle Infrastructure Cess queries
        cess_keywords = ["infrastructure cess", "cess act", "cess", "infrastructure tax", "balochistan infrastructure", "cess 2021", "amended 2024", "cess rates", "cess collection", "cess penalties"]
        if self.fuzzy_match_keywords(processed_query, cess_keywords):
            return self.handle_infrastructure_cess_query(processed_query)
        
        
        # Default response for unmatched queries
        return {
            "response": "I understand you're asking about BRA services. Could you please be more specific? You can ask about:\n• Registration procedures\n• Tax rates and payments\n• Contact information\n• E-services\n• Appeals and complaints\n\nOr visit our FAQ section for detailed information.",
            "type": "clarification",
            "confidence": "low"
        }

# Initialize chatbot
bra_bot = BRAChatbot()

@app.route('/')
def index():
    """Serve the main chatbot interface"""
    return render_template('index.html')

@app.route('/api/chat', methods=['POST'])
def chat_api():
    """API endpoint for chatbot interactions"""
    try:
        data = request.get_json()
        user_message = data.get('message', '').strip()
        
        if not user_message:
            return jsonify({
                "error": "Message cannot be empty",
                "status": "error"
            }), 400
        
        # Generate response
        print(f"Processing message: {user_message}")
        response_data = bra_bot.generate_response(user_message)
        print(f"Response data: {response_data}")
        
        return jsonify({
            "message": response_data["response"],
            "type": response_data["type"],
            "confidence": response_data.get("confidence", "medium"),
            "timestamp": datetime.now().isoformat(),
            "status": "success"
        })
        
    except Exception as e:
        print(f"Error in chat API: {e}")
        import traceback
        traceback.print_exc()
        return jsonify({
            "error": f"An error occurred: {str(e)}",
            "status": "error"
        }), 500

@app.route('/api/faqs', methods=['GET'])
def get_faqs():
    """API endpoint to get all FAQs"""
    return jsonify({
        "faqs": bra_bot.kb.faqs,
        "status": "success"
    })

@app.route('/api/contact', methods=['GET'])
def get_contact():
    """API endpoint to get contact information"""
    return jsonify({
        "contact": bra_bot.kb.bra_info["contact"],
        "status": "success"
    })

@app.route('/api/suggestions', methods=['GET'])
def get_suggestions():
    """API endpoint to get dynamic topic suggestions"""
    # Get query parameter for context
    context = request.args.get('context', '').lower()
    
    # Define suggestion categories
    suggestions = {
        'default': [
            {"text": "What is BRA?", "icon": "fas fa-info-circle", "query": "What is BRA?"},
            {"text": "How to register?", "icon": "fas fa-user-plus", "query": "How to register for BRA services?"},
            {"text": "Tax rates", "icon": "fas fa-percentage", "query": "What are the tax rates?"},
            {"text": "Contact info", "icon": "fas fa-phone", "query": "Contact information"},
            {"text": "Download forms", "icon": "fas fa-file-download", "query": "Download forms"}
        ],
        'registration': [
            {"text": "e-Registration process", "icon": "fas fa-user-plus", "query": "How to register for BRA services?"},
            {"text": "Registration requirements", "icon": "fas fa-list-check", "query": "What are the registration requirements?"},
            {"text": "Check registration status", "icon": "fas fa-search", "query": "How to check registration status?"},
            {"text": "Registration forms", "icon": "fas fa-file-alt", "query": "Download registration forms"},
            {"text": "NTN registration", "icon": "fas fa-id-card", "query": "How to register with NTN?"}
        ],
        'tax': [
            {"text": "Sales tax rates", "icon": "fas fa-percentage", "query": "What are the sales tax rates?"},
            {"text": "How to pay taxes", "icon": "fas fa-credit-card", "query": "How to pay BRA taxes online?"},
            {"text": "Tax calculation", "icon": "fas fa-calculator", "query": "How to calculate tax?"},
            {"text": "Tax returns", "icon": "fas fa-file-invoice", "query": "How to file tax returns?"},
            {"text": "Tax exemptions", "icon": "fas fa-shield-alt", "query": "What are the tax exemptions?"}
        ],
        'services': [
            {"text": "e-Services overview", "icon": "fas fa-laptop", "query": "What are BRA e-Services?"},
            {"text": "Online payment", "icon": "fas fa-credit-card", "query": "How to pay taxes online?"},
            {"text": "Return filing", "icon": "fas fa-file-invoice", "query": "How to file returns?"},
            {"text": "Taxpayer search", "icon": "fas fa-search", "query": "How to check taxpayer status?"},
            {"text": "Withholding agent", "icon": "fas fa-handshake", "query": "How to become a withholding agent?"}
        ],
        'forms': [
            {"text": "Registration form", "icon": "fas fa-file-alt", "query": "Download registration form"},
            {"text": "Return form", "icon": "fas fa-file-invoice", "query": "Download return form"},
            {"text": "Payment challan", "icon": "fas fa-credit-card", "query": "Download payment challan"},
            {"text": "All forms", "icon": "fas fa-folder", "query": "What forms are available for download?"},
            {"text": "Form instructions", "icon": "fas fa-question-circle", "query": "How to fill forms?"}
        ],
        'contact': [
            {"text": "Phone number", "icon": "fas fa-phone", "query": "What is BRA phone number?"},
            {"text": "Email address", "icon": "fas fa-envelope", "query": "What is BRA email?"},
            {"text": "Office address", "icon": "fas fa-map-marker-alt", "query": "Where is BRA located?"},
            {"text": "Office hours", "icon": "fas fa-clock", "query": "What are BRA office hours?"},
            {"text": "Support", "icon": "fas fa-headset", "query": "How to get BRA support?"}
        ],
        'infrastructure': [
            {"text": "Infrastructure Cess", "icon": "fas fa-road", "query": "What is Infrastructure Cess?"},
            {"text": "Cess rates", "icon": "fas fa-percentage", "query": "What are the cess rates?"},
            {"text": "Cess payment", "icon": "fas fa-credit-card", "query": "How to pay Infrastructure Cess?"},
            {"text": "Cess exemptions", "icon": "fas fa-shield-alt", "query": "What are cess exemptions?"},
            {"text": "Cess act", "icon": "fas fa-gavel", "query": "Infrastructure Cess Act details"}
        ],
        'management': [
            {"text": "Chairperson", "icon": "fas fa-user-tie", "query": "Who is the BRA Chairperson?"},
            {"text": "Management team", "icon": "fas fa-users", "query": "What is BRA management structure?"},
            {"text": "Commissioners", "icon": "fas fa-user-shield", "query": "Who are the BRA commissioners?"},
            {"text": "Department heads", "icon": "fas fa-user-cog", "query": "BRA department heads"},
            {"text": "Contact management", "icon": "fas fa-address-book", "query": "How to contact BRA management?"}
        ]
    }
    
    # Determine context based on query
    if any(word in context for word in ['register', 'registration', 'sign up', 'enroll']):
        selected_suggestions = suggestions['registration']
    elif any(word in context for word in ['tax', 'payment', 'rate', 'percentage']):
        selected_suggestions = suggestions['tax']
    elif any(word in context for word in ['service', 'online', 'e-service', 'portal']):
        selected_suggestions = suggestions['services']
    elif any(word in context for word in ['form', 'download', 'document', 'bst']):
        selected_suggestions = suggestions['forms']
    elif any(word in context for word in ['contact', 'phone', 'email', 'address', 'location']):
        selected_suggestions = suggestions['contact']
    elif any(word in context for word in ['infrastructure', 'cess', 'road', 'development']):
        selected_suggestions = suggestions['infrastructure']
    elif any(word in context for word in ['management', 'chairperson', 'commissioner', 'team']):
        selected_suggestions = suggestions['management']
    else:
        selected_suggestions = suggestions['default']
    
    return jsonify({
        "suggestions": selected_suggestions,
        "context": context,
        "status": "success"
    })

@app.route('/download/<document_id>')
def download_document(document_id):
    """API endpoint to download BRA documents"""
    try:
        documents = bra_bot.kb.bra_info["documents"]
        
        if document_id not in documents:
            return jsonify({
                "error": "Document not found",
                "status": "error"
            }), 404
        
        document = documents[document_id]
        filename = document["file"]
        
        # Check if file exists
        file_path = os.path.join(app.static_folder, 'assets', filename)
        if not os.path.exists(file_path):
            return jsonify({
                "error": "File not found on server",
                "status": "error"
            }), 404
        
        return send_from_directory(
            os.path.join(app.static_folder, 'assets'),
            filename,
            as_attachment=True,
            download_name=filename
        )
        
    except Exception as e:
        return jsonify({
            "error": f"Download failed: {str(e)}",
            "status": "error"
        }), 500

@app.route('/api/documents', methods=['GET'])
def get_documents():
    """API endpoint to get list of available documents"""
    return jsonify({
        "documents": bra_bot.kb.bra_info["documents"],
        "status": "success"
    })

if __name__ == '__main__':
    print("🤖 Starting BRA Chatbot Server...")
    print("🌐 Access the chatbot at: http://localhost:5001")
    print("📡 API available at: http://localhost:5001/api/chat")
    app.run(debug=True, host='0.0.0.0', port=5001)
