// BRA Chatbot JavaScript
class BRAChatbot {
    constructor() {
        this.apiUrl = '/api/chat';
        this.messageInput = document.getElementById('messageInput');
        this.sendButton = document.getElementById('sendButton');
        this.chatMessages = document.getElementById('chatMessages');
        this.typingIndicator = document.getElementById('typingIndicator');
        this.suggestions = document.getElementById('suggestions');
        
        this.isTyping = false;
        this.conversationId = this.generateConversationId();
        
        this.init();
    }
    
    init() {
        // Set initial time
        document.getElementById('initialTime').textContent = this.formatTime(new Date());
        
        // Event listeners
        this.messageInput.addEventListener('keypress', (e) => {
            if (e.key === 'Enter' && !e.shiftKey) {
                e.preventDefault();
                this.sendMessage();
            }
        });
        
        this.sendButton.addEventListener('click', () => this.sendMessage());
        
        // Auto-resize input
        this.messageInput.addEventListener('input', () => {
            this.updateSendButton();
        });
        
        // Initial state
        this.updateSendButton();
        
        // Focus input
        this.messageInput.focus();
    }
    
    generateConversationId() {
        return 'conv_' + Date.now() + '_' + Math.random().toString(36).substr(2, 9);
    }
    
    updateSendButton() {
        const hasText = this.messageInput.value.trim().length > 0;
        this.sendButton.disabled = !hasText || this.isTyping;
        
        if (hasText && !this.isTyping) {
            this.sendButton.style.background = 'linear-gradient(135deg, #3b82f6 0%, #1e3a8a 100%)';
        } else {
            this.sendButton.style.background = '#9ca3af';
        }
    }
    
    formatMessageWithLinks(text) {
        // Convert markdown-style links to HTML links
        let formattedText = text.replace(/\[([^\]]+)\]\(([^)]+)\)/g, (match, linkText, url) => {
            if (url.startsWith('/download/')) {
                const documentId = url.split('/').pop();
                // Clean up the link text by removing extra emojis and formatting
                const cleanLinkText = linkText.replace(/📥\s*/, '').replace(/DOWNLOAD\s*/i, 'DOWNLOAD ');
                return `<a href="${url}" class="download-link" download="${documentId}_Form.txt" target="_blank">
                    <i class="fas fa-download"></i> ${cleanLinkText}
                </a>`;
            } else if (url.startsWith('http')) {
                return `<a href="${url}" target="_blank" rel="noopener noreferrer" style="color: #3b82f6; text-decoration: underline;">${linkText}</a>`;
            }
            return `<a href="${url}" target="_blank" rel="noopener noreferrer">${linkText}</a>`;
        });
        
        // Convert **bold** text to HTML
        formattedText = formattedText.replace(/\*\*([^*]+)\*\*/g, '<strong>$1</strong>');
        
        // Convert bullet points
        formattedText = formattedText.replace(/^• (.+)$/gm, '<li style="margin-left: 20px;">$1</li>');
        
        // Convert line breaks
        formattedText = formattedText.replace(/\n/g, '<br>');
        
        return formattedText;
    }
    
    async sendMessage() {
        const message = this.messageInput.value.trim();
        if (!message || this.isTyping) return;
        
        // Add user message to chat
        this.addMessage(message, 'user');
        
        // Clear input
        this.messageInput.value = '';
        this.updateSendButton();
        
        // Show typing indicator
        this.showTypingIndicator();
        
        try {
            // Send message to backend
            const response = await fetch(this.apiUrl, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    message: message,
                    conversation_id: this.conversationId
                })
            });
            
            const data = await response.json();
            
            if (data.status === 'success') {
                // Store the user message for later use
                const userMessage = message;
                
                // Simulate typing delay for better UX
                setTimeout(() => {
                    this.hideTypingIndicator();
                    this.addMessage(data.message, 'bot', data.type);
                    
                    // Update dynamic suggestions based on user query
                    console.log('🔄 Triggering suggestions update for:', userMessage);
                    loadDynamicSuggestions(userMessage);
                }, 1000 + Math.random() * 1000); // 1-2 seconds delay
            } else {
                this.hideTypingIndicator();
                this.addMessage('Sorry, I encountered an error. Please try again.', 'bot', 'error');
            }
        } catch (error) {
            console.error('Error sending message:', error);
            this.hideTypingIndicator();
            this.addMessage('Sorry, I\'m having trouble connecting. Please check your internet connection and try again.', 'bot', 'error');
        }
    }
    
    addMessage(text, sender, type = 'message') {
        const messageDiv = document.createElement('div');
        messageDiv.className = `message ${sender}-message`;
        
        const avatarDiv = document.createElement('div');
        avatarDiv.className = 'message-avatar';
        
        if (sender === 'bot') {
            avatarDiv.innerHTML = '<i class="fas fa-robot"></i>';
        } else {
            avatarDiv.innerHTML = '<i class="fas fa-user"></i>';
        }
        
        const contentDiv = document.createElement('div');
        contentDiv.className = 'message-content';
        
        const textDiv = document.createElement('div');
        textDiv.className = 'message-text';
        
        // Handle markdown-style links and make them clickable
        if (text.includes('](/download/') || text.includes('[📥 DOWNLOAD')) {
            textDiv.innerHTML = this.formatMessageWithLinks(text);
        } else {
            textDiv.textContent = text;
        }
        
        // Add special styling for different message types
        if (type === 'error') {
            textDiv.style.background = 'linear-gradient(135deg, #dc2626 0%, #b91c1c 100%)';
            textDiv.style.color = 'white';
        } else if (type === 'contact') {
            textDiv.style.background = 'linear-gradient(135deg, #10b981 0%, #059669 100%)';
            textDiv.style.color = 'white';
        } else if (type === 'document') {
            textDiv.style.background = 'linear-gradient(135deg, #f9fafb 0%, #ffffff 100%)';
            textDiv.style.borderLeft = '4px solid #10b981';
            textDiv.style.padding = '16px 20px';
        }
        
        const timeDiv = document.createElement('div');
        timeDiv.className = 'message-time';
        timeDiv.innerHTML = `<span>${this.formatTime(new Date())}</span>`;
        
        contentDiv.appendChild(textDiv);
        contentDiv.appendChild(timeDiv);
        
        messageDiv.appendChild(avatarDiv);
        messageDiv.appendChild(contentDiv);
        
        this.chatMessages.appendChild(messageDiv);
        this.scrollToBottom();
        
        // Add message animation
        messageDiv.style.opacity = '0';
        messageDiv.style.transform = 'translateY(20px)';
        setTimeout(() => {
            messageDiv.style.transition = 'all 0.3s ease';
            messageDiv.style.opacity = '1';
            messageDiv.style.transform = 'translateY(0)';
        }, 100);
    }
    
    showTypingIndicator() {
        this.isTyping = true;
        this.updateSendButton();
        this.typingIndicator.style.display = 'flex';
        this.scrollToBottom();
    }
    
    hideTypingIndicator() {
        this.isTyping = false;
        this.updateSendButton();
        this.typingIndicator.style.display = 'none';
    }
    
    scrollToBottom() {
        setTimeout(() => {
            this.chatMessages.scrollTop = this.chatMessages.scrollHeight;
        }, 100);
    }
    
    formatTime(date) {
        return date.toLocaleTimeString('en-US', {
            hour: '2-digit',
            minute: '2-digit',
            hour12: true
        });
    }
    
    
    sendQuickMessage(message) {
        this.messageInput.value = message;
        this.sendMessage();
    }
}

// Global functions for HTML onclick events
function sendQuickMessage(message) {
    if (window.chatbot) {
        window.chatbot.sendQuickMessage(message);
    }
}

function clearChat() {
    if (confirm('Are you sure you want to clear the chat history?')) {
        const chatMessages = document.getElementById('chatMessages');
        chatMessages.innerHTML = `
            <div class="message bot-message">
                <div class="message-avatar">
                    <i class="fas fa-robot"></i>
                </div>
                <div class="message-content">
                    <div class="message-text">
                        Chat cleared! How can I help you with BRA services today?
                    </div>
                    <div class="message-time">
                        <span>${new Date().toLocaleTimeString('en-US', {
                            hour: '2-digit',
                            minute: '2-digit',
                            hour12: true
                        })}</span>
                    </div>
                </div>
            </div>
        `;
        
        // Reset suggestions
        if (window.chatbot) {
            window.chatbot.updateSuggestionButtons([
                'Sales tax rate',
                'Registration procedure',
                'File return'
            ]);
        }
    }
}

function minimizeChat() {
    const chatContainer = document.querySelector('.chatbot-container');
    const isMinimized = chatContainer.style.display === 'none';
    
    if (isMinimized) {
        chatContainer.style.display = 'block';
        chatContainer.style.animation = 'fadeInUp 0.3s ease';
    } else {
        chatContainer.style.display = 'none';
    }
}

// Enhanced error handling
window.addEventListener('error', (event) => {
    console.error('JavaScript error:', event.error);
});

// Network status monitoring
window.addEventListener('online', () => {
    console.log('Connection restored');
    if (window.chatbot) {
        window.chatbot.addMessage('Connection restored. You can continue chatting!', 'bot', 'success');
    }
});

window.addEventListener('offline', () => {
    console.log('Connection lost');
    if (window.chatbot) {
        window.chatbot.addMessage('Connection lost. Please check your internet connection.', 'bot', 'error');
    }
});

// Keyboard shortcuts
document.addEventListener('keydown', (e) => {
    // Ctrl/Cmd + L to clear chat
    if ((e.ctrlKey || e.metaKey) && e.key === 'l') {
        e.preventDefault();
        clearChat();
    }
    
    // Escape to minimize chat
    if (e.key === 'Escape') {
        minimizeChat();
    }
});

// Responsive design helpers
function handleResize() {
    const chatMessages = document.getElementById('chatMessages');
    if (window.innerWidth <= 480) {
        chatMessages.style.height = '300px';
    } else if (window.innerWidth <= 768) {
        chatMessages.style.height = '350px';
    } else {
        chatMessages.style.height = '400px';
    }
}

window.addEventListener('resize', handleResize);

// Load dynamic suggestions based on context
function loadDynamicSuggestions(context = '') {
    console.log('🔄 Loading dynamic suggestions for context:', context);
    
    fetch(`/api/suggestions?context=${encodeURIComponent(context)}`)
        .then(response => {
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            return response.json();
        })
        .then(data => {
            console.log('📡 Suggestions API response:', data);
            if (data.status === 'success' && data.suggestions) {
                updateTopicButtons(data.suggestions);
                updateInputSuggestions(data.suggestions);
                console.log('✅ Suggestions updated successfully');
            } else {
                console.warn('⚠️ API returned unsuccessful response, loading defaults');
                loadDefaultSuggestions();
            }
        })
        .catch(error => {
            console.error('❌ Error loading suggestions:', error);
            // Load default suggestions on error
            loadDefaultSuggestions();
        });
}

// Update topic buttons with dynamic suggestions
function updateTopicButtons(suggestions) {
    const topicButtons = document.getElementById('topic-buttons');
    if (!topicButtons) {
        console.error('❌ Topic buttons element not found');
        return;
    }
    
    console.log('🔄 Updating topic buttons with:', suggestions);
    topicButtons.innerHTML = '';
    
    if (suggestions && suggestions.length > 0) {
        suggestions.forEach(suggestion => {
            const button = document.createElement('button');
            button.className = 'topic-btn';
            button.innerHTML = `<i class="${suggestion.icon || 'fas fa-question'}"></i> ${suggestion.text}`;
            button.onclick = () => sendQuickMessage(suggestion.query);
            topicButtons.appendChild(button);
        });
        console.log('✅ Topic buttons updated with', suggestions.length, 'buttons');
    } else {
        console.warn('⚠️ No suggestions provided for topic buttons');
    }
}

// Update input suggestions with dynamic suggestions
function updateInputSuggestions(suggestions) {
    const inputSuggestions = document.getElementById('input-suggestions');
    if (!inputSuggestions) {
        console.error('❌ Input suggestions element not found');
        return;
    }
    
    console.log('🔄 Updating input suggestions with:', suggestions);
    inputSuggestions.innerHTML = '';
    
    if (suggestions && suggestions.length > 0) {
        // Take first 3 suggestions for input area
        const inputSuggestionsList = suggestions.slice(0, 3);
        
        inputSuggestionsList.forEach(suggestion => {
            const button = document.createElement('button');
            button.textContent = suggestion.text;
            button.onclick = () => sendQuickMessage(suggestion.query);
            inputSuggestions.appendChild(button);
        });
        
        console.log('✅ Input suggestions updated with', inputSuggestionsList.length, 'buttons');
    } else {
        console.warn('⚠️ No suggestions provided for input suggestions');
    }
}

// Load default suggestions as fallback
function loadDefaultSuggestions() {
    const defaultSuggestions = [
        {"text": "What is BRA?", "icon": "fas fa-info-circle", "query": "What is BRA?"},
        {"text": "How to register?", "icon": "fas fa-user-plus", "query": "How to register for BRA services?"},
        {"text": "Tax rates", "icon": "fas fa-percentage", "query": "What are the tax rates?"},
        {"text": "Contact info", "icon": "fas fa-phone", "query": "Contact information"},
        {"text": "Download forms", "icon": "fas fa-file-download", "query": "Download forms"}
    ];
    updateTopicButtons(defaultSuggestions);
    updateInputSuggestions(defaultSuggestions);
}

// Initialize chatbot when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    console.log('🤖 Initializing BRA Chatbot...');
    
    // Initialize chatbot
    window.chatbot = new BRAChatbot();
    
    // Load dynamic suggestions
    loadDynamicSuggestions();
    
    // Also load default suggestions as backup
    setTimeout(() => {
        const inputSuggestions = document.getElementById('input-suggestions');
        if (inputSuggestions && inputSuggestions.children.length === 0) {
            console.log('🔄 Loading default suggestions as backup');
            loadDefaultSuggestions();
        }
    }, 2000);
    
    // Add global function for testing
    window.testSuggestions = function(context) {
        console.log('🧪 Testing suggestions with context:', context);
        loadDynamicSuggestions(context);
    };
    
    // Handle initial responsive design
    handleResize();
    
    // Add some welcome animations
    setTimeout(() => {
        const welcomeCard = document.querySelector('.welcome-card');
        if (welcomeCard) {
            welcomeCard.style.animation = 'fadeInUp 0.6s ease';
        }
        
        const chatContainer = document.querySelector('.chatbot-container');
        if (chatContainer) {
            chatContainer.style.animation = 'fadeInUp 0.8s ease';
        }
        
        const infoCards = document.querySelectorAll('.card');
        infoCards.forEach((card, index) => {
            setTimeout(() => {
                card.style.animation = 'fadeInUp 0.6s ease';
            }, index * 200);
        });
    }, 500);
    
    console.log('✅ BRA Chatbot initialized successfully!');
});

// Service Worker registration for offline support (optional)
if ('serviceWorker' in navigator) {
    window.addEventListener('load', () => {
        // You can add a service worker here for offline functionality
        console.log('Service Worker support detected');
    });
}

// Analytics and user interaction tracking (optional)
function trackUserInteraction(action, data) {
    // You can implement analytics tracking here
    console.log('User interaction:', action, data);
}

// Export for testing purposes
if (typeof module !== 'undefined' && module.exports) {
    module.exports = { BRAChatbot };
}
